const { Sequelize } = require('sequelize');
const path = require('path');

// Initialize Sequelize with SQLite
const sequelize = new Sequelize({
  dialect: 'sqlite',
  storage: path.join(__dirname, 'database.sqlite'),
  logging: false
});

async function checkDatabase() {
  try {
    console.log('Checking database connection...');
    await sequelize.authenticate();
    console.log('✅ Database connection successful!\n');

    // Get all table names
    const [results] = await sequelize.query(`
      SELECT name FROM sqlite_master 
      WHERE type='table' 
      ORDER BY name;
    `);

    console.log('📋 Tables in database:');
    results.forEach(table => {
      console.log(`  - ${table.name}`);
    });

    // Check specifically for Bkash tables
    const bkashTables = results.filter(table => 
      table.name.toLowerCase().includes('bkash')
    );

    console.log('\n🏦 Bkash-related tables:');
    if (bkashTables.length === 0) {
      console.log('  ❌ No Bkash tables found!');
    } else {
      bkashTables.forEach(table => {
        console.log(`  ✅ ${table.name}`);
      });
    }

    // Check BkashAgentAccounts table structure if it exists
    const agentTableExists = results.some(table => 
      table.name === 'BkashAgentAccounts'
    );

    if (agentTableExists) {
      console.log('\n📊 BkashAgentAccounts table structure:');
      const [columns] = await sequelize.query(`
        PRAGMA table_info(BkashAgentAccounts);
      `);
      columns.forEach(col => {
        console.log(`  - ${col.name}: ${col.type} ${col.notnull ? 'NOT NULL' : ''} ${col.pk ? 'PRIMARY KEY' : ''}`);
      });

      // Check if there are any records
      const [count] = await sequelize.query(`
        SELECT COUNT(*) as count FROM BkashAgentAccounts;
      `);
      console.log(`\n📈 Records in BkashAgentAccounts: ${count[0].count}`);
    }

  } catch (error) {
    console.error('❌ Database error:', error.message);
  } finally {
    await sequelize.close();
  }
}

checkDatabase();