const db = require('../models');
const User = db.users;
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');

// Secret key for JWT
const JWT_SECRET = process.env.JWT_SECRET || 'pos-system-secret-key';

// Register a new user
exports.register = async (req, res) => {
  try {
    // Validate request
    if (!req.body.username || !req.body.password || !req.body.email || !req.body.fullName) {
      return res.status(400).send({
        message: "Username, password, email, and full name are required!"
      });
    }

    // Check if username or email already exists
    const existingUser = await User.findOne({
      where: {
        [db.Sequelize.Op.or]: [
          { username: req.body.username },
          { email: req.body.email }
        ]
      }
    });

    if (existingUser) {
      return res.status(400).send({
        message: "Username or email already exists!"
      });
    }

    // Hash password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(req.body.password, salt);

    // Create user
    const user = {
      username: req.body.username,
      email: req.body.email,
      password: hashedPassword,
      fullName: req.body.fullName,
      role: req.body.role || 'cashier'
    };

    // Save user in the database
    const data = await User.create(user);

    // Remove password from response
    const { password, ...userWithoutPassword } = data.toJSON();

    res.status(201).send(userWithoutPassword);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the user."
    });
  }
};

// User login
exports.login = async (req, res) => {
  try {
    // Validate request
    const loginField = req.body.username || req.body.email;
    if (!loginField || !req.body.password) {
      return res.status(400).send({
        message: "Username/email and password are required!"
      });
    }

    // Find user by username or email
    const user = await User.findOne({
      where: {
        [db.Sequelize.Op.or]: [
          { username: loginField },
          { email: loginField }
        ]
      }
    });

    if (!user) {
      return res.status(404).send({
        message: "User not found!"
      });
    }

    // Check if user is active
    if (user.status !== 'active') {
      return res.status(403).send({
        message: "Account is inactive. Please contact administrator."
      });
    }

    // Validate password
    const validPassword = await bcrypt.compare(req.body.password, user.password);
    if (!validPassword) {
      return res.status(401).send({
        message: "Invalid password!"
      });
    }

    // Update last login time
    await User.update(
      { lastLogin: new Date() },
      { where: { id: user.id } }
    );

    // Generate JWT token
    const token = jwt.sign(
      { id: user.id, username: user.username, role: user.role },
      JWT_SECRET,
      { expiresIn: '24h' }
    );

    // Remove password from response
    const { password, ...userWithoutPassword } = user.toJSON();

    res.send({
      ...userWithoutPassword,
      token
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred during login."
    });
  }
};

// Get current user profile
exports.profile = async (req, res) => {
  try {
    const user = await User.findByPk(req.userId, {
      attributes: { exclude: ['password'] }
    });

    if (!user) {
      return res.status(404).send({
        message: "User not found!"
      });
    }

    res.send(user);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving user profile."
    });
  }
};