const db = require('../models');
const BkashAgentAccount = db.bkashAgentAccounts;
const BkashTransaction = db.bkashTransactions;
const { Op } = require('sequelize');

// Create or update agent account for a specific date
exports.createOrUpdate = async (req, res) => {
  try {
    const { 
      date, 
      openingBalance, 
      closingBalance, 
      srReceived, 
      srGiven, 
      bankDeposit, 
      bankWithdraw, 
      b2bSend,
      b2bReceived,
      amountReceived,
      bkashAccountNumberId,
      notes,
      commissionRate,
      totalTransaction,
      totalCommission
    } = req.body;

    // Validate required fields
    if (!date || openingBalance === undefined) {
      return res.status(400).send({
        message: "Date and opening balance are required!"
      });
    }

    // Check if account for this date already exists
    let account = await BkashAgentAccount.findOne({ where: { date } });

    // Prepare update data with automatic calculation logic
    const updatedOpeningBalance = openingBalance !== undefined ? parseFloat(openingBalance) : (account ? parseFloat(account.openingBalance) : 0);
    const updatedClosingBalance = closingBalance !== undefined ? parseFloat(closingBalance) : (account ? account.closingBalance : null);
    const updatedCommissionRate = commissionRate !== undefined ? parseFloat(commissionRate) : (account ? parseFloat(account.commissionRate || 0.40) : 0.40);
    
    let updatedTotalTransaction = totalTransaction;
    let updatedTotalCommission = totalCommission;

    // Auto-calculate if closing balance is provided and total transaction is not manually set
    if (updatedClosingBalance !== null && totalTransaction === undefined && updatedOpeningBalance > 0) {
      const calculatedTransaction = updatedOpeningBalance - updatedClosingBalance;
      updatedTotalTransaction = calculatedTransaction > 0 ? calculatedTransaction : 0;
      updatedTotalCommission = (updatedTotalTransaction * updatedCommissionRate) / 100;
    }

    if (account) {
      // Update existing account
      await account.update({
        openingBalance: openingBalance || account.openingBalance,
        closingBalance: closingBalance !== undefined ? closingBalance : account.closingBalance,
        srReceived: srReceived || account.srReceived,
        srGiven: srGiven || account.srGiven,
        bankDeposit: bankDeposit || account.bankDeposit,
        bankWithdraw: bankWithdraw || account.bankWithdraw,
        b2bSend: b2bSend || account.b2bSend,
        b2bReceived: b2bReceived || account.b2bReceived,
        amountReceived: amountReceived || account.amountReceived,
        bkashAccountNumberId: bkashAccountNumberId !== undefined ? bkashAccountNumberId : account.bkashAccountNumberId,
        notes: notes !== undefined ? notes : account.notes,
        commissionRate: updatedCommissionRate,
        totalTransaction: updatedTotalTransaction !== undefined ? updatedTotalTransaction : account.totalTransaction,
        totalCommission: updatedTotalCommission !== undefined ? updatedTotalCommission : account.totalCommission
      });
    } else {
      // Create new account
      account = await BkashAgentAccount.create({
        date,
        openingBalance,
        closingBalance,
        srReceived: srReceived || 0,
        srGiven: srGiven || 0,
        bankDeposit: bankDeposit || 0,
        bankWithdraw: bankWithdraw || 0,
        b2bSend: b2bSend || 0,
        b2bReceived: b2bReceived || 0,
        amountReceived: amountReceived || 0,
        bkashAccountNumberId,
        notes,
        commissionRate: updatedCommissionRate,
        totalTransaction: updatedTotalTransaction,
        totalCommission: updatedTotalCommission
      });
    }

    // Create transaction records if amounts are provided
    const transactions = [];
    
    if (srReceived && srReceived > 0) {
      transactions.push({
        date,
        type: 'sr_received',
        accountType: 'agent',
        amount: srReceived,
        agentAccountId: account.id,
        description: 'Money received from Bkash SR'
      });
    }

    if (srGiven && srGiven > 0) {
      transactions.push({
        date,
        type: 'sr_given',
        accountType: 'agent',
        amount: srGiven,
        agentAccountId: account.id,
        description: 'Money given to Bkash SR'
      });
    }

    if (bankDeposit && bankDeposit > 0) {
      transactions.push({
        date,
        type: 'bank_deposit',
        accountType: 'agent',
        amount: bankDeposit,
        agentAccountId: account.id,
        description: 'Money deposited to bank'
      });
    }

    if (bankWithdraw && bankWithdraw > 0) {
      transactions.push({
        date,
        type: 'bank_withdraw',
        accountType: 'agent',
        amount: bankWithdraw,
        agentAccountId: account.id,
        description: 'Money withdrawn from bank'
      });
    }

    if (b2bSend && b2bSend > 0) {
      transactions.push({
        date,
        type: 'b2b_send',
        accountType: 'agent',
        amount: b2bSend,
        agentAccountId: account.id,
        description: 'Money sent via B2B transaction'
      });
    }

    if (b2bReceived && b2bReceived > 0) {
      transactions.push({
        date,
        type: 'b2b_received',
        accountType: 'agent',
        amount: b2bReceived,
        agentAccountId: account.id,
        description: 'Money received via B2B transaction'
      });
    }

    if (amountReceived && amountReceived > 0) {
      transactions.push({
        date,
        type: 'amount_received',
        accountType: 'agent',
        amount: amountReceived,
        agentAccountId: account.id,
        description: 'Amount received from customers/transactions'
      });
    }

    // Bulk create transactions
    if (transactions.length > 0) {
      await BkashTransaction.bulkCreate(transactions);
    }

    // Fetch updated account with transactions
    const updatedAccount = await BkashAgentAccount.findByPk(account.id, {
      include: [{
        model: BkashTransaction,
        where: { agentAccountId: account.id },
        required: false
      }]
    });

    res.send(updatedAccount);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating/updating the agent account."
    });
  }
};

// Get all agent accounts
exports.findAll = (req, res) => {
  const { startDate, endDate, limit } = req.query;
  
  let condition = {};
  if (startDate && endDate) {
    condition.date = {
      [Op.between]: [startDate, endDate]
    };
  } else if (startDate) {
    condition.date = {
      [Op.gte]: startDate
    };
  } else if (endDate) {
    condition.date = {
      [Op.lte]: endDate
    };
  }

  BkashAgentAccount.findAll({
    where: condition,
    include: [{
      model: BkashTransaction,
      required: false
    }],
    order: [['date', 'DESC']],
    limit: limit ? parseInt(limit) : undefined
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving agent accounts."
      });
    });
};

// Get agent account by date
exports.findByDate = (req, res) => {
  const date = req.params.date;

  BkashAgentAccount.findOne({
    where: { date },
    include: [{
      model: BkashTransaction,
      required: false
    }]
  })
    .then(data => {
      if (data) {
        res.send(data);
      } else {
        res.status(404).send({
          message: `Cannot find agent account for date ${date}.`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving agent account for date " + date
      });
    });
};

// Get today's agent account
exports.findToday = (req, res) => {
  const today = new Date();
  const todayString = today.getFullYear() + '-' + 
                     String(today.getMonth() + 1).padStart(2, '0') + '-' + 
                     String(today.getDate()).padStart(2, '0');

  BkashAgentAccount.findOne({
    where: { date: todayString },
    include: [{
      model: BkashTransaction,
      required: false
    }]
  })
    .then(data => {
      if (data) {
        res.send(data);
      } else {
        // Create today's account with default values
        BkashAgentAccount.create({
          date: todayString,
          openingBalance: 0
        })
          .then(newAccount => {
            res.send(newAccount);
          })
          .catch(err => {
            res.status(500).send({
              message: err.message || "Error creating today's agent account."
            });
          });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving today's agent account"
      });
    });
};

// Update closing balance and complete the day
exports.closeDay = async (req, res) => {
  try {
    const { date, closingBalance, notes, b2bSend, b2bReceived, nextDayOpening, bkashAccountNumberId } = req.body;

    if (!date || closingBalance === undefined) {
      return res.status(400).send({
        message: "Date and closing balance are required!"
      });
    }

    const account = await BkashAgentAccount.findOne({ where: { date } });

    if (!account) {
      return res.status(404).send({
        message: `Cannot find agent account for date ${date}.`
      });
    }

    // Update current day's account
    await account.update({
      closingBalance,
      b2bSend: b2bSend || 0,
      b2bReceived: b2bReceived || 0,
      notes: notes !== undefined ? notes : account.notes
    });

    // Create B2B transactions if they exist
    const transactions = [];
    
    if (b2bSend && b2bSend > 0) {
      transactions.push({
        date,
        type: 'b2b_send',
        accountType: 'agent',
        amount: b2bSend,
        agentAccountId: account.id,
        description: 'Money sent via B2B transaction (Day Close)'
      });
    }

    if (b2bReceived && b2bReceived > 0) {
      transactions.push({
        date,
        type: 'b2b_received',
        accountType: 'agent',
        amount: b2bReceived,
        agentAccountId: account.id,
        description: 'Money received via B2B transaction (Day Close)'
      });
    }

    // Create transactions if any
    if (transactions.length > 0) {
      await BkashTransaction.bulkCreate(transactions);
    }

    // Create next day's opening balance if it doesn't exist
    if (nextDayOpening !== undefined && bkashAccountNumberId) {
      const nextDate = new Date(date);
      nextDate.setDate(nextDate.getDate() + 1);
      const nextDateStr = nextDate.getFullYear() + '-' + 
                         String(nextDate.getMonth() + 1).padStart(2, '0') + '-' + 
                         String(nextDate.getDate()).padStart(2, '0');

      const existingNextDay = await BkashAgentAccount.findOne({ 
        where: { date: nextDateStr } 
      });

      if (!existingNextDay) {
        await BkashAgentAccount.create({
          date: nextDateStr,
          openingBalance: nextDayOpening,
          closingBalance: nextDayOpening, // Initialize with opening balance
          bkashAccountNumberId: bkashAccountNumberId,
          b2bSend: 0,
          b2bReceived: 0,
          bankDeposit: 0,
          bankWithdraw: 0,
          amountReceived: 0
        });
      } else {
        // Update existing next day's opening balance
        await existingNextDay.update({
          openingBalance: nextDayOpening
        });
      }
    }

    // Fetch updated account with calculated profit
    const updatedAccount = await BkashAgentAccount.findByPk(account.id, {
      include: [{
        model: BkashTransaction,
        required: false
      }]
    });

    res.send(updatedAccount);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while closing the day."
    });
  }
};

// Get summary statistics
exports.getSummary = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    const today = new Date();
    const todayString = today.getFullYear() + '-' + 
                       String(today.getMonth() + 1).padStart(2, '0') + '-' + 
                       String(today.getDate()).padStart(2, '0');
    
    let dateCondition = {};
    if (startDate && endDate) {
      dateCondition.date = { [Op.between]: [startDate, endDate] };
    } else {
      // Default to current month
      const firstDayDate = new Date(new Date().getFullYear(), new Date().getMonth(), 1);
      const firstDay = firstDayDate.getFullYear() + '-' + 
                      String(firstDayDate.getMonth() + 1).padStart(2, '0') + '-' + 
                      String(firstDayDate.getDate()).padStart(2, '0');
      dateCondition.date = { [Op.gte]: firstDay };
    }

    // Get total profit for the period
    const totalProfit = await BkashAgentAccount.sum('dailyProfit', {
      where: {
        ...dateCondition,
        dailyProfit: { [Op.not]: null }
      }
    });

    // Get today's account
    const todayAccount = await BkashAgentAccount.findOne({
      where: { date: todayString }
    });

    // Get total transactions for the period
    const totalTransactions = await BkashTransaction.count({
      where: {
        ...dateCondition,
        accountType: 'agent'
      }
    });

    // Get completed days count
    const completedDays = await BkashAgentAccount.count({
      where: {
        ...dateCondition,
        isCompleted: true
      }
    });

    res.send({
      totalProfit: totalProfit || 0,
      todayAccount: todayAccount || null,
      totalTransactions,
      completedDays,
      period: { startDate, endDate }
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving summary."
    });
  }
};

// Delete agent account
exports.delete = (req, res) => {
  const date = req.params.date;

  BkashAgentAccount.destroy({
    where: { date }
  })
    .then(num => {
      if (num == 1) {
        res.send({
          message: "Agent account was deleted successfully!"
        });
      } else {
        res.send({
          message: `Cannot delete agent account for date ${date}. Maybe it was not found!`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Could not delete agent account for date " + date
      });
    });
};