const db = require('../models');
const BkashPersonalAccount = db.bkashPersonalAccounts;
const BkashTransaction = db.bkashTransactions;
const { Op } = require('sequelize');

// Create or update personal account for a specific date
exports.createOrUpdate = async (req, res) => {
  try {
    const { 
      date, 
      openingBalance, 
      closingBalance, 
      totalSent, 
      totalReceived, 
      notes 
    } = req.body;

    // Validate required fields
    if (!date || openingBalance === undefined) {
      return res.status(400).send({
        message: "Date and opening balance are required!"
      });
    }

    // Check if account for this date already exists
    let account = await BkashPersonalAccount.findOne({ where: { date } });

    if (account) {
      // Update existing account
      await account.update({
        openingBalance: openingBalance || account.openingBalance,
        closingBalance: closingBalance !== undefined ? closingBalance : account.closingBalance,
        totalSent: totalSent || account.totalSent,
        totalReceived: totalReceived || account.totalReceived,
        notes: notes !== undefined ? notes : account.notes
      });
    } else {
      // Create new account
      account = await BkashPersonalAccount.create({
        date,
        openingBalance,
        closingBalance,
        totalSent: totalSent || 0,
        totalReceived: totalReceived || 0,
        notes
      });
    }

    // Create transaction records if amounts are provided
    const transactions = [];
    
    if (totalSent && totalSent > 0) {
      transactions.push({
        date,
        type: 'sent',
        accountType: 'personal',
        amount: totalSent,
        personalAccountId: account.id,
        description: 'Money sent from personal Bkash'
      });
    }

    if (totalReceived && totalReceived > 0) {
      transactions.push({
        date,
        type: 'received',
        accountType: 'personal',
        amount: totalReceived,
        personalAccountId: account.id,
        description: 'Money received to personal Bkash'
      });
    }

    // Bulk create transactions
    if (transactions.length > 0) {
      await BkashTransaction.bulkCreate(transactions);
    }

    // Fetch updated account with transactions
    const updatedAccount = await BkashPersonalAccount.findByPk(account.id, {
      include: [{
        model: BkashTransaction,
        where: { personalAccountId: account.id },
        required: false
      }]
    });

    res.send(updatedAccount);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating/updating the personal account."
    });
  }
};

// Get all personal accounts
exports.findAll = (req, res) => {
  const { startDate, endDate, limit } = req.query;
  
  let condition = {};
  if (startDate && endDate) {
    condition.date = {
      [Op.between]: [startDate, endDate]
    };
  } else if (startDate) {
    condition.date = {
      [Op.gte]: startDate
    };
  } else if (endDate) {
    condition.date = {
      [Op.lte]: endDate
    };
  }

  BkashPersonalAccount.findAll({
    where: condition,
    include: [{
      model: BkashTransaction,
      required: false
    }],
    order: [['date', 'DESC']],
    limit: limit ? parseInt(limit) : undefined
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving personal accounts."
      });
    });
};

// Get personal account by date
exports.findByDate = (req, res) => {
  const date = req.params.date;

  BkashPersonalAccount.findOne({
    where: { date },
    include: [{
      model: BkashTransaction,
      required: false
    }]
  })
    .then(data => {
      if (data) {
        res.send(data);
      } else {
        res.status(404).send({
          message: `Cannot find personal account for date ${date}.`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving personal account for date " + date
      });
    });
};

// Get today's personal account
exports.findToday = (req, res) => {
  const today = new Date().toISOString().split('T')[0];

  BkashPersonalAccount.findOne({
    where: { date: today },
    include: [{
      model: BkashTransaction,
      required: false
    }]
  })
    .then(data => {
      if (data) {
        res.send(data);
      } else {
        // Create today's account with default values
        BkashPersonalAccount.create({
          date: today,
          openingBalance: 0
        })
          .then(newAccount => {
            res.send(newAccount);
          })
          .catch(err => {
            res.status(500).send({
              message: err.message || "Error creating today's personal account."
            });
          });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving today's personal account"
      });
    });
};

// Add a transaction to personal account
exports.addTransaction = async (req, res) => {
  try {
    const { date, type, amount, description } = req.body;

    if (!date || !type || !amount) {
      return res.status(400).send({
        message: "Date, type, and amount are required!"
      });
    }

    if (!['sent', 'received'].includes(type)) {
      return res.status(400).send({
        message: "Transaction type must be 'sent' or 'received'!"
      });
    }

    // Find or create personal account for the date
    let account = await BkashPersonalAccount.findOne({ where: { date } });
    
    if (!account) {
      account = await BkashPersonalAccount.create({
        date,
        openingBalance: 0
      });
    }

    // Create transaction
    const transaction = await BkashTransaction.create({
      date,
      type,
      accountType: 'personal',
      amount,
      personalAccountId: account.id,
      description: description || `Money ${type} via personal Bkash`
    });

    // Update account totals
    if (type === 'sent') {
      await account.update({
        totalSent: (account.totalSent || 0) + parseFloat(amount)
      });
    } else if (type === 'received') {
      await account.update({
        totalReceived: (account.totalReceived || 0) + parseFloat(amount)
      });
    }

    // Fetch updated account with all transactions
    const updatedAccount = await BkashPersonalAccount.findByPk(account.id, {
      include: [{
        model: BkashTransaction,
        required: false
      }]
    });

    res.send({
      transaction,
      account: updatedAccount
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while adding the transaction."
    });
  }
};

// Update closing balance and complete the day
exports.closeDay = async (req, res) => {
  try {
    const { date, closingBalance, notes } = req.body;

    if (!date || closingBalance === undefined) {
      return res.status(400).send({
        message: "Date and closing balance are required!"
      });
    }

    const account = await BkashPersonalAccount.findOne({ where: { date } });

    if (!account) {
      return res.status(404).send({
        message: `Cannot find personal account for date ${date}.`
      });
    }

    await account.update({
      closingBalance,
      notes: notes !== undefined ? notes : account.notes
    });

    // Fetch updated account with calculated net change
    const updatedAccount = await BkashPersonalAccount.findByPk(account.id, {
      include: [{
        model: BkashTransaction,
        required: false
      }]
    });

    res.send(updatedAccount);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while closing the day."
    });
  }
};

// Get summary statistics
exports.getSummary = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    const today = new Date().toISOString().split('T')[0];
    
    let dateCondition = {};
    if (startDate && endDate) {
      dateCondition.date = { [Op.between]: [startDate, endDate] };
    } else {
      // Default to current month
      const firstDay = new Date(new Date().getFullYear(), new Date().getMonth(), 1).toISOString().split('T')[0];
      dateCondition.date = { [Op.gte]: firstDay };
    }

    // Get total net change for the period
    const totalNetChange = await BkashPersonalAccount.sum('netChange', {
      where: {
        ...dateCondition,
        netChange: { [Op.not]: null }
      }
    });

    // Get today's account
    const todayAccount = await BkashPersonalAccount.findOne({
      where: { date: today }
    });

    // Get total transactions for the period
    const totalTransactions = await BkashTransaction.count({
      where: {
        ...dateCondition,
        accountType: 'personal'
      }
    });

    // Get total sent and received
    const totalSent = await BkashPersonalAccount.sum('totalSent', {
      where: dateCondition
    });

    const totalReceived = await BkashPersonalAccount.sum('totalReceived', {
      where: dateCondition
    });

    // Get completed days count
    const completedDays = await BkashPersonalAccount.count({
      where: {
        ...dateCondition,
        isCompleted: true
      }
    });

    res.send({
      totalNetChange: totalNetChange || 0,
      totalSent: totalSent || 0,
      totalReceived: totalReceived || 0,
      todayAccount: todayAccount || null,
      totalTransactions,
      completedDays,
      period: { startDate, endDate }
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving summary."
    });
  }
};

// Get transactions for a specific date
exports.getTransactions = (req, res) => {
  const date = req.params.date;

  BkashTransaction.findAll({
    where: { 
      date,
      accountType: 'personal'
    },
    order: [['createdAt', 'DESC']]
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving transactions."
      });
    });
};

// Delete personal account
exports.delete = (req, res) => {
  const date = req.params.date;

  BkashPersonalAccount.destroy({
    where: { date }
  })
    .then(num => {
      if (num == 1) {
        res.send({
          message: "Personal account was deleted successfully!"
        });
      } else {
        res.send({
          message: `Cannot delete personal account for date ${date}. Maybe it was not found!`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Could not delete personal account for date " + date
      });
    });
};