const db = require("../models");
const BkashTransaction = db.bkashTransactions;
const BkashAgentAccount = db.bkashAgentAccounts;
const BkashPersonalAccount = db.bkashPersonalAccounts;
const BkashMerchantAccount = db.bkashMerchantAccounts;
const BkashAccountNumber = db.bkashAccountNumbers;
const { Op } = require("sequelize");

// Get transactions with filters
exports.getTransactions = async (req, res) => {
  try {
    const { 
      accountType, 
      accountNumberId, 
      startDate, 
      endDate, 
      limit = 50,
      offset = 0 
    } = req.query;

    let whereCondition = {};
    
    // Filter by account type
    if (accountType) {
      whereCondition.accountType = accountType;
    }
    
    // Filter by date range
    if (startDate && endDate) {
      whereCondition.date = {
        [Op.between]: [startDate, endDate]
      };
    } else if (startDate) {
      whereCondition.date = {
        [Op.gte]: startDate
      };
    } else if (endDate) {
      whereCondition.date = {
        [Op.lte]: endDate
      };
    }

    // Filter by specific account number
    if (accountNumberId) {
      const accountNumber = await BkashAccountNumber.findByPk(accountNumberId);
      if (accountNumber) {
        if (accountNumber.accountType === 'agent') {
          // Find agent accounts for this account number
          const agentAccounts = await BkashAgentAccount.findAll({
            where: { bkashAccountNumberId: accountNumberId }
          });
          const agentAccountIds = agentAccounts.map(acc => acc.id);
          whereCondition.agentAccountId = { [Op.in]: agentAccountIds };
        } else if (accountNumber.accountType === 'personal') {
          // Find personal accounts for this account number
          const personalAccounts = await BkashPersonalAccount.findAll({
            where: { bkashAccountNumberId: accountNumberId }
          });
          const personalAccountIds = personalAccounts.map(acc => acc.id);
          whereCondition.personalAccountId = { [Op.in]: personalAccountIds };
        } else if (accountNumber.accountType === 'merchant') {
          // Find merchant accounts for this account number
          const merchantAccounts = await BkashMerchantAccount.findAll({
            where: { bkashAccountNumberId: accountNumberId }
          });
          const merchantAccountIds = merchantAccounts.map(acc => acc.id);
          whereCondition.merchantAccountId = { [Op.in]: merchantAccountIds };
        }
      }
    }

    const transactions = await BkashTransaction.findAll({
      where: whereCondition,
      include: [
        {
          model: BkashAccountNumber,
          as: 'bkashAccountNumber',
          required: false
        }
      ],
      order: [['date', 'DESC'], ['time', 'DESC']],
      limit: parseInt(limit),
      offset: parseInt(offset)
    });

    res.json(transactions);
  } catch (error) {
    console.error('Error getting transactions:', error);
    res.status(500).json({ 
      message: "Error retrieving transactions",
      error: error.message 
    });
  }
};

// Get account summary (balance and profit) for a specific account and date range
exports.getAccountSummary = async (req, res) => {
  try {
    const { accountType, accountNumberId, startDate, endDate } = req.query;

    if (!accountType || !accountNumberId || !startDate || !endDate) {
      return res.status(400).json({
        message: "accountType, accountNumberId, startDate, and endDate are required"
      });
    }

    let accountModel;
    let accountIdField;
    
    switch (accountType) {
      case 'agent':
        accountModel = BkashAgentAccount;
        accountIdField = 'agentAccountId';
        break;
      case 'personal':
        accountModel = BkashPersonalAccount;
        accountIdField = 'personalAccountId';
        break;
      case 'merchant':
        accountModel = BkashMerchantAccount;
        accountIdField = 'merchantAccountId';
        break;
      default:
        return res.status(400).json({ message: "Invalid account type" });
    }

    // Get accounts for the date range
    let accounts = await accountModel.findAll({
      where: {
        bkashAccountNumberId: accountNumberId,
        date: {
          [Op.between]: [startDate, endDate]
        }
      },
      order: [['date', 'ASC']]
    });

    // If no accounts found with specific accountNumberId, try to find legacy accounts (where bkashAccountNumberId is null)
    if (accounts.length === 0) {
      accounts = await accountModel.findAll({
        where: {
          [Op.or]: [
            { bkashAccountNumberId: null },
            { bkashAccountNumberId: accountNumberId }
          ],
          date: {
            [Op.between]: [startDate, endDate]
          }
        },
        order: [['date', 'ASC']]
      });
    }

    if (accounts.length === 0) {
      return res.json({
        openingBalance: 0,
        closingBalance: 0,
        totalReceived: 0,
        totalGiven: 0,
        dailyProfit: 0,
        accountCount: 0
      });
    }

    // Calculate summary
    const firstAccount = accounts[0];
    const lastAccount = accounts[accounts.length - 1];
    
    let totalReceived = 0;
    let totalGiven = 0;
    
    accounts.forEach(account => {
      if (accountType === 'agent') {
        totalReceived += parseFloat(account.srReceived || 0) + parseFloat(account.bankWithdraw || 0);
        totalGiven += parseFloat(account.srGiven || 0) + parseFloat(account.bankDeposit || 0);
      } else if (accountType === 'personal') {
        totalReceived += parseFloat(account.totalReceived || 0);
        totalGiven += parseFloat(account.totalSent || 0);
      } else if (accountType === 'merchant') {
        totalReceived += parseFloat(account.totalSales || 0);
        totalGiven += parseFloat(account.totalRefunds || 0) + parseFloat(account.totalWithdrawals || 0) + parseFloat(account.merchantFee || 0);
      }
    });

    const openingBalance = parseFloat(firstAccount.openingBalance || 0);
    const closingBalance = parseFloat(lastAccount.closingBalance || lastAccount.openingBalance || 0);
    const dailyProfit = closingBalance - openingBalance + totalReceived - totalGiven;

    res.json({
      openingBalance,
      closingBalance,
      totalReceived,
      totalGiven,
      dailyProfit,
      accountCount: accounts.length
    });

  } catch (error) {
    console.error('Error getting account summary:', error);
    res.status(500).json({ 
      message: "Error retrieving account summary",
      error: error.message 
    });
  }
};

// Get combined report for all accounts
exports.getCombinedReport = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;

    if (!startDate || !endDate) {
      return res.status(400).json({
        message: "startDate and endDate are required"
      });
    }

    // Get all account numbers
    const accountNumbers = await BkashAccountNumber.findAll({
      where: { isActive: true }
    });

    const combinedReport = {
      totalAccounts: accountNumbers.length,
      totalOpeningBalance: 0,
      totalClosingBalance: 0,
      totalReceived: 0,
      totalGiven: 0,
      totalProfit: 0,
      accountDetails: []
    };

    // Process each account
    for (const accountNumber of accountNumbers) {
      let accountModel;
      let accountIdField;
      
      switch (accountNumber.accountType) {
        case 'agent':
          accountModel = BkashAgentAccount;
          break;
        case 'personal':
          accountModel = BkashPersonalAccount;
          break;
        case 'merchant':
          accountModel = BkashMerchantAccount;
          break;
        default:
          continue;
      }

      // Get accounts for this account number in the date range
      const accounts = await accountModel.findAll({
        where: {
          bkashAccountNumberId: accountNumber.id,
          date: {
            [Op.between]: [startDate, endDate]
          }
        },
        order: [['date', 'ASC']]
      });

      if (accounts.length > 0) {
        const firstAccount = accounts[0];
        const lastAccount = accounts[accounts.length - 1];
        
        let totalReceived = 0;
        let totalGiven = 0;
        
        accounts.forEach(account => {
          if (accountNumber.accountType === 'agent') {
            totalReceived += parseFloat(account.srReceived || 0) + parseFloat(account.bankWithdraw || 0);
            totalGiven += parseFloat(account.srGiven || 0) + parseFloat(account.bankDeposit || 0);
          } else if (accountNumber.accountType === 'personal') {
            totalReceived += parseFloat(account.totalReceived || 0);
            totalGiven += parseFloat(account.totalSent || 0);
          } else if (accountNumber.accountType === 'merchant') {
            totalReceived += parseFloat(account.totalSales || 0);
            totalGiven += parseFloat(account.totalRefunds || 0) + parseFloat(account.totalWithdrawals || 0) + parseFloat(account.merchantFee || 0);
          }
        });

        const openingBalance = parseFloat(firstAccount.openingBalance || 0);
        const closingBalance = parseFloat(lastAccount.closingBalance || lastAccount.openingBalance || 0);
        const dailyProfit = closingBalance - openingBalance + totalReceived - totalGiven;

        const accountDetail = {
          accountNumber: accountNumber.accountNumber,
          accountType: accountNumber.accountType,
          openingBalance,
          closingBalance,
          totalReceived,
          totalGiven,
          dailyProfit
        };

        combinedReport.accountDetails.push(accountDetail);
        combinedReport.totalOpeningBalance += openingBalance;
        combinedReport.totalClosingBalance += closingBalance;
        combinedReport.totalReceived += totalReceived;
        combinedReport.totalGiven += totalGiven;
        combinedReport.totalProfit += dailyProfit;
      }
    }

    res.json(combinedReport);

  } catch (error) {
    console.error('Error getting combined report:', error);
    res.status(500).json({ 
      message: "Error retrieving combined report",
      error: error.message 
    });
  }
};

// Save opening/closing balance
exports.saveBalance = async (req, res) => {
  try {
    const { accountType, accountNumberId, balanceType, amount, date } = req.body;

    // Validate required fields
    if (!accountType || !accountNumberId || !balanceType || !amount || !date) {
      return res.status(400).json({
        success: false,
        message: "All fields are required"
      });
    }

    // Validate balance type
    if (!['opening', 'closing'].includes(balanceType)) {
      return res.status(400).json({
        success: false,
        message: "Balance type must be 'opening' or 'closing'"
      });
    }

    // Validate amount
    if (isNaN(amount) || amount < 0) {
      return res.status(400).json({
        success: false,
        message: "Amount must be a valid positive number"
      });
    }

    // Get the account number details
    const accountNumber = await BkashAccountNumber.findByPk(accountNumberId);
    if (!accountNumber) {
      return res.status(404).json({
        success: false,
        message: "Account number not found"
      });
    }

    // Update the actual account balance based on account type
    let updatedAccount = null;
    const balanceField = balanceType === 'opening' ? 'openingBalance' : 'closingBalance';
    
    switch (accountType) {
      case 'agent':
        // Find or create agent account for this date
        const [agentAccount] = await BkashAgentAccount.findOrCreate({
          where: { 
            date: date,
            bkashAccountNumberId: accountNumberId 
          },
          defaults: {
            date: date,
            bkashAccountNumberId: accountNumberId,
            openingBalance: balanceType === 'opening' ? amount : 0,
            closingBalance: balanceType === 'closing' ? amount : 0
          }
        });
        
        // Update the balance
        await agentAccount.update({
          [balanceField]: amount
        });
        updatedAccount = agentAccount;
        break;
        
      case 'personal':
        // Find or create personal account for this date
        const [personalAccount] = await BkashPersonalAccount.findOrCreate({
          where: { 
            date: date,
            bkashAccountNumberId: accountNumberId 
          },
          defaults: {
            date: date,
            bkashAccountNumberId: accountNumberId,
            openingBalance: balanceType === 'opening' ? amount : 0,
            closingBalance: balanceType === 'closing' ? amount : 0
          }
        });
        
        // Update the balance
        await personalAccount.update({
          [balanceField]: amount
        });
        updatedAccount = personalAccount;
        break;
        
      case 'merchant':
        // Find or create merchant account for this date
        const [merchantAccount] = await BkashMerchantAccount.findOrCreate({
          where: { 
            date: date,
            bkashAccountNumberId: accountNumberId 
          },
          defaults: {
            date: date,
            bkashAccountNumberId: accountNumberId,
            openingBalance: balanceType === 'opening' ? amount : 0,
            closingBalance: balanceType === 'closing' ? amount : 0
          }
        });
        
        // Update the balance
        await merchantAccount.update({
          [balanceField]: amount
        });
        updatedAccount = merchantAccount;
        break;
        
      default:
        return res.status(400).json({
          success: false,
          message: "Invalid account type"
        });
    }

    // Create transaction record for audit trail
    const transactionData = {
      accountType: accountType,
      date: date,
      time: new Date().toLocaleTimeString(),
      type: balanceType === 'opening' ? 'opening_balance' : 'closing_balance',
      amount: amount,
      description: `${balanceType.charAt(0).toUpperCase() + balanceType.slice(1)} balance for ${date}`,
      reference: `${balanceType.toUpperCase()}_${Date.now()}`,
      accountNumber: accountNumber.accountNumber
    };

    // Save the transaction
    const transaction = await BkashTransaction.create(transactionData);

    res.json({
      success: true,
      message: `${balanceType.charAt(0).toUpperCase() + balanceType.slice(1)} balance saved successfully`,
      transaction: transaction,
      account: updatedAccount
    });

  } catch (error) {
    console.error('Error saving balance:', error);
    res.status(500).json({
      success: false,
      message: "Error saving balance",
      error: error.message
    });
  }
};