const db = require("../models");
const Customer = db.customers;
const Op = db.Sequelize.Op;

// Create and Save a new Customer
exports.create = (req, res) => {
  // Validate request
  if (!req.body.name) {
    res.status(400).send({
      message: "Customer name cannot be empty!"
    });
    return;
  }

  // Create a Customer
  const customer = {
    name: req.body.name,
    email: req.body.email,
    phone: req.body.phone,
    address: req.body.address,
    dueBalance: req.body.dueBalance || 0.00,
    creditLimit: req.body.creditLimit || 0.00,
    totalPurchases: req.body.totalPurchases || 0.00,
    lastPurchaseDate: req.body.lastPurchaseDate,
    notes: req.body.notes,
    isActive: req.body.isActive !== undefined ? req.body.isActive : true
  };

  // Save Customer in the database
  Customer.create(customer)
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while creating the Customer."
      });
    });
};

// Retrieve all Customers from the database
exports.findAll = (req, res) => {
  const name = req.query.name;
  const condition = name ? { name: { [Op.like]: `%${name}%` } } : null;

  Customer.findAll({ 
    where: condition,
    order: [['createdAt', 'DESC']]
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving customers."
      });
    });
};

// Find a single Customer with an id
exports.findOne = (req, res) => {
  const id = req.params.id;

  Customer.findByPk(id)
    .then(data => {
      if (data) {
        res.send(data);
      } else {
        res.status(404).send({
          message: `Cannot find Customer with id=${id}.`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error retrieving Customer with id=" + id
      });
    });
};

// Update a Customer by the id in the request
exports.update = (req, res) => {
  const id = req.params.id;

  Customer.update(req.body, {
    where: { id: id }
  })
    .then(num => {
      if (num == 1) {
        res.send({
          message: "Customer was updated successfully."
        });
      } else {
        res.status(404).send({
          message: `Cannot update Customer with id=${id}. Maybe Customer was not found or req.body is empty!`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Error updating Customer with id=" + id
      });
    });
};

// Delete a Customer with the specified id in the request
exports.delete = (req, res) => {
  const id = req.params.id;

  Customer.destroy({
    where: { id: id }
  })
    .then(num => {
      if (num == 1) {
        res.send({
          message: "Customer was deleted successfully!"
        });
      } else {
        res.status(404).send({
          message: `Cannot delete Customer with id=${id}. Maybe Customer was not found!`
        });
      }
    })
    .catch(err => {
      res.status(500).send({
        message: "Could not delete Customer with id=" + id
      });
    });
};

// Delete all Customers from the database
exports.deleteAll = (req, res) => {
  Customer.destroy({
    where: {},
    truncate: false
  })
    .then(nums => {
      res.send({ message: `${nums} Customers were deleted successfully!` });
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while removing all customers."
      });
    });
};

// Find all active Customers
exports.findAllActive = (req, res) => {
  Customer.findAll({ 
    where: { isActive: true },
    order: [['name', 'ASC']]
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving customers."
      });
    });
};

// Get customers with due balance
exports.findWithDue = (req, res) => {
  Customer.findAll({ 
    where: { 
      dueBalance: { [Op.gt]: 0 },
      isActive: true 
    },
    order: [['dueBalance', 'DESC']]
  })
    .then(data => {
      res.send(data);
    })
    .catch(err => {
      res.status(500).send({
        message: err.message || "Some error occurred while retrieving customers with due balance."
      });
    });
};

// Update customer due balance
exports.updateDueBalance = (req, res) => {
  const id = req.params.id;
  const { amount, operation } = req.body; // operation: 'add' or 'subtract'

  Customer.findByPk(id)
    .then(customer => {
      if (!customer) {
        return res.status(404).send({
          message: `Customer with id=${id} not found.`
        });
      }

      let newBalance = parseFloat(customer.dueBalance);
      if (operation === 'add') {
        newBalance += parseFloat(amount);
      } else if (operation === 'subtract') {
        newBalance -= parseFloat(amount);
      }

      // Ensure balance doesn't go negative
      newBalance = Math.max(0, newBalance);

      return Customer.update(
        { dueBalance: newBalance },
        { where: { id: id } }
      );
    })
    .then(() => {
      res.send({
        message: "Customer due balance updated successfully."
      });
    })
    .catch(err => {
      res.status(500).send({
        message: "Error updating customer due balance with id=" + id
      });
    });
};

// Get customer purchase history (this would need a sales table relationship)
exports.getPurchaseHistory = (req, res) => {
  const id = req.params.id;
  
  // For now, return empty array - this would be implemented when sales are linked to customers
  res.send([]);
};