const db = require('../models');
const DSODelivery = db.dsoDeliveries;
const DSOAllocation = db.dsoAllocations;
const { Op } = require('sequelize');

// Create a new DSO delivery
exports.createDelivery = async (req, res) => {
  try {
    const {
      date,
      dsoName,
      receivedBy,
      amountPaid,
      cardsReceivedTotal,
      cardTypes,
      simsReceived,
      kitsReceived,
      notes,
      createdBy
    } = req.body;

    // Validation
    if (!dsoName || !receivedBy || !createdBy) {
      return res.status(400).send({
        message: "DSO name, received by, and created by are required"
      });
    }

    if (amountPaid < 0) {
      return res.status(400).send({
        message: "Amount paid must be non-negative"
      });
    }

    if (cardsReceivedTotal === 0 && simsReceived === 0 && kitsReceived === 0) {
      return res.status(400).send({
        message: "At least one of Cards, SIMs, or Kits must be received (quantity > 0)"
      });
    }

    // Validate card types if provided and filled
    if (cardTypes && cardTypes.length > 0) {
      // Filter out completely empty card type entries (both type and quantity are empty)
      const filledCardTypes = cardTypes.filter(ct => 
        (ct.type && ct.type.trim() !== '') && (ct.quantity && ct.quantity > 0)
      );
      
      // Only validate if user has actually filled in complete card type information
      if (filledCardTypes.length > 0) {
        const totalFromTypes = filledCardTypes.reduce((sum, cardType) => sum + (cardType.quantity || 0), 0);
        if (totalFromTypes !== cardsReceivedTotal) {
          return res.status(400).send({
            message: "Sum of card type quantities must equal total cards received"
          });
        }
        
        // Check for incomplete entries (entries that have either type OR quantity but not both)
        const incompleteEntries = cardTypes.filter(ct => 
          (ct.type && ct.type.trim() !== '' && (!ct.quantity || ct.quantity <= 0)) ||
          (ct.quantity && ct.quantity > 0 && (!ct.type || ct.type.trim() === ''))
        );
        
        if (incompleteEntries.length > 0) {
          return res.status(400).send({
            message: "All card types must have both a name and positive quantity"
          });
        }
      }
    }

    const deliveryData = {
      dsoName,
      receivedBy,
      amountPaid: parseFloat(amountPaid) || 0,
      cardsReceivedTotal: parseInt(cardsReceivedTotal) || 0,
      cardTypes: cardTypes || [],
      simsReceived: parseInt(simsReceived) || 0,
      kitsReceived: parseInt(kitsReceived) || 0,
      notes,
      createdBy
    };

    // Only set date if provided and valid
    if (date && date.trim() !== '') {
      deliveryData.date = date;
    }

    const delivery = await DSODelivery.create(deliveryData);

    res.status(201).send({
      message: "DSO delivery created successfully",
      delivery
    });
  } catch (error) {
    console.error('Error creating DSO delivery:', error);
    res.status(500).send({
      message: "Error creating DSO delivery",
      error: error.message
    });
  }
};

// Get all deliveries with optional filters
exports.getDeliveries = async (req, res) => {
  try {
    const { 
      startDate, 
      endDate, 
      dsoName, 
      page = 1, 
      limit = 10 
    } = req.query;

    const offset = (page - 1) * limit;
    const whereClause = {};

    if (startDate && endDate) {
      whereClause.date = {
        [Op.between]: [startDate, endDate]
      };
    } else if (startDate) {
      whereClause.date = {
        [Op.gte]: startDate
      };
    } else if (endDate) {
      whereClause.date = {
        [Op.lte]: endDate
      };
    }

    if (dsoName) {
      whereClause.dsoName = {
        [Op.like]: `%${dsoName}%`
      };
    }

    const { count, rows } = await DSODelivery.findAndCountAll({
      where: whereClause,
      include: [{
        model: DSOAllocation,
        required: false
      }],
      order: [['date', 'DESC'], ['createdAt', 'DESC']],
      limit: parseInt(limit),
      offset: parseInt(offset)
    });

    res.send({
      deliveries: rows,
      totalCount: count,
      currentPage: parseInt(page),
      totalPages: Math.ceil(count / limit)
    });
  } catch (error) {
    console.error('Error getting DSO deliveries:', error);
    res.status(500).send({
      message: "Error retrieving DSO deliveries",
      error: error.message
    });
  }
};

// Get delivery by ID
exports.getDeliveryById = async (req, res) => {
  try {
    const { id } = req.params;

    const delivery = await DSODelivery.findByPk(id, {
      include: [{
        model: DSOAllocation,
        required: false
      }]
    });

    if (!delivery) {
      return res.status(404).send({
        message: "DSO delivery not found"
      });
    }

    res.send(delivery);
  } catch (error) {
    console.error('Error getting DSO delivery:', error);
    res.status(500).send({
      message: "Error retrieving DSO delivery",
      error: error.message
    });
  }
};

// Update delivery (manager role only)
exports.updateDelivery = async (req, res) => {
  try {
    const { id } = req.params;
    const { editedBy, ...updateData } = req.body;

    if (!editedBy) {
      return res.status(400).send({
        message: "Edited by is required"
      });
    }

    const delivery = await DSODelivery.findByPk(id);
    if (!delivery) {
      return res.status(404).send({
        message: "DSO delivery not found"
      });
    }

    await delivery.update({
      ...updateData,
      editedBy
    });

    res.send({
      message: "DSO delivery updated successfully",
      delivery
    });
  } catch (error) {
    console.error('Error updating DSO delivery:', error);
    res.status(500).send({
      message: "Error updating DSO delivery",
      error: error.message
    });
  }
};

// Create allocation from delivery
exports.createAllocation = async (req, res) => {
  try {
    const {
      deliveryId,
      date,
      cardsAllocated,
      cardTypesAllocated,
      simsAllocated,
      kitsAllocated,
      allocatedBy,
      notes
    } = req.body;

    if (!allocatedBy) {
      return res.status(400).send({
        message: "Allocated by is required"
      });
    }

    if (cardsAllocated === 0 && simsAllocated === 0 && kitsAllocated === 0) {
      return res.status(400).send({
        message: "At least one of Cards, SIMs, or Kits must be allocated (quantity > 0)"
      });
    }

    // Validate against delivery if deliveryId provided
    if (deliveryId) {
      const delivery = await DSODelivery.findByPk(deliveryId);
      if (!delivery) {
        return res.status(404).send({
          message: "DSO delivery not found"
        });
      }

      // Check if allocation exceeds received quantities
      if (cardsAllocated > delivery.cardsReceivedTotal) {
        return res.status(400).send({
          message: "Cards allocated cannot exceed cards received"
        });
      }
      if (simsAllocated > delivery.simsReceived) {
        return res.status(400).send({
          message: "SIMs allocated cannot exceed SIMs received"
        });
      }
      if (kitsAllocated > delivery.kitsReceived) {
        return res.status(400).send({
          message: "Kits allocated cannot exceed kits received"
        });
      }
    }

    // Validate card types allocation
    if (cardTypesAllocated && cardTypesAllocated.length > 0) {
      const totalFromTypes = cardTypesAllocated.reduce((sum, cardType) => sum + (cardType.quantity || 0), 0);
      if (totalFromTypes !== cardsAllocated) {
        return res.status(400).send({
          message: "Sum of allocated card type quantities must equal total cards allocated"
        });
      }
    }

    const allocation = await DSOAllocation.create({
      deliveryId,
      date: date || new Date().toISOString().split('T')[0],
      cardsAllocated: parseInt(cardsAllocated) || 0,
      cardTypesAllocated: cardTypesAllocated || [],
      simsAllocated: parseInt(simsAllocated) || 0,
      kitsAllocated: parseInt(kitsAllocated) || 0,
      allocatedBy,
      notes
    });

    res.status(201).send({
      message: "DSO allocation created successfully",
      allocation
    });
  } catch (error) {
    console.error('Error creating DSO allocation:', error);
    res.status(500).send({
      message: "Error creating DSO allocation",
      error: error.message
    });
  }
};

// Get allocations with optional filters
exports.getAllocations = async (req, res) => {
  try {
    const { 
      startDate, 
      endDate, 
      deliveryId,
      page = 1, 
      limit = 10 
    } = req.query;

    const offset = (page - 1) * limit;
    const whereClause = {};

    if (startDate && endDate) {
      whereClause.date = {
        [Op.between]: [startDate, endDate]
      };
    }

    if (deliveryId) {
      whereClause.deliveryId = deliveryId;
    }

    const { count, rows } = await DSOAllocation.findAndCountAll({
      where: whereClause,
      include: [{
        model: DSODelivery,
        required: false
      }],
      order: [['date', 'DESC'], ['allocatedAt', 'DESC']],
      limit: parseInt(limit),
      offset: parseInt(offset)
    });

    res.send({
      allocations: rows,
      totalCount: count,
      currentPage: parseInt(page),
      totalPages: Math.ceil(count / limit)
    });
  } catch (error) {
    console.error('Error getting DSO allocations:', error);
    res.status(500).send({
      message: "Error retrieving DSO allocations",
      error: error.message
    });
  }
};

// Get today's allocations summary
exports.getTodayAllocations = async (req, res) => {
  try {
    const today = new Date().toISOString().split('T')[0];

    const allocations = await DSOAllocation.findAll({
      where: {
        date: today,
        status: 'active'
      },
      include: [{
        model: DSODelivery,
        required: false
      }]
    });

    const summary = allocations.reduce((acc, allocation) => {
      acc.totalCardsAllocated += allocation.cardsAllocated;
      acc.totalSimsAllocated += allocation.simsAllocated;
      acc.totalKitsAllocated += allocation.kitsAllocated;
      acc.totalCardsSold += allocation.cardsSold;
      acc.totalSimsSold += allocation.simsSold;
      acc.totalKitsSold += allocation.kitsSold;
      return acc;
    }, {
      totalCardsAllocated: 0,
      totalSimsAllocated: 0,
      totalKitsAllocated: 0,
      totalCardsSold: 0,
      totalSimsSold: 0,
      totalKitsSold: 0
    });

    res.send({
      date: today,
      summary,
      allocations
    });
  } catch (error) {
    console.error('Error getting today\'s allocations:', error);
    res.status(500).send({
      message: "Error retrieving today's allocations",
      error: error.message
    });
  }
};

// Record sale from allocated stock
exports.recordSale = async (req, res) => {
  try {
    const {
      allocationId,
      itemType, // 'cards', 'sims', 'kits'
      quantity,
      cardType // optional, for card sales
    } = req.body;

    if (!allocationId || !itemType || !quantity) {
      return res.status(400).send({
        message: "Allocation ID, item type, and quantity are required"
      });
    }

    const allocation = await DSOAllocation.findByPk(allocationId);
    if (!allocation) {
      return res.status(404).send({
        message: "DSO allocation not found"
      });
    }

    let updateData = {};

    switch (itemType) {
      case 'cards':
        if (allocation.cardsSold + quantity > allocation.cardsAllocated) {
          return res.status(400).send({
            message: "Cannot sell more cards than allocated"
          });
        }
        updateData.cardsSold = allocation.cardsSold + quantity;
        
        // Update card types sold if cardType provided
        if (cardType) {
          const cardTypesSold = [...(allocation.cardTypesSold || [])];
          const existingTypeIndex = cardTypesSold.findIndex(ct => ct.type === cardType);
          if (existingTypeIndex >= 0) {
            cardTypesSold[existingTypeIndex].quantity += quantity;
          } else {
            cardTypesSold.push({ type: cardType, quantity });
          }
          updateData.cardTypesSold = cardTypesSold;
        }
        break;
        
      case 'sims':
        if (allocation.simsSold + quantity > allocation.simsAllocated) {
          return res.status(400).send({
            message: "Cannot sell more SIMs than allocated"
          });
        }
        updateData.simsSold = allocation.simsSold + quantity;
        break;
        
      case 'kits':
        if (allocation.kitsSold + quantity > allocation.kitsAllocated) {
          return res.status(400).send({
            message: "Cannot sell more kits than allocated"
          });
        }
        updateData.kitsSold = allocation.kitsSold + quantity;
        break;
        
      default:
        return res.status(400).send({
          message: "Invalid item type. Must be 'cards', 'sims', or 'kits'"
        });
    }

    await allocation.update(updateData);

    res.send({
      message: "Sale recorded successfully",
      allocation
    });
  } catch (error) {
    console.error('Error recording sale:', error);
    res.status(500).send({
      message: "Error recording sale",
      error: error.message
    });
  }
};

// Get daily DSO activity report
exports.getDailyReport = async (req, res) => {
  try {
    const { 
      startDate, 
      endDate, 
      dsoName 
    } = req.query;

    const today = new Date().toISOString().split('T')[0];
    const reportStartDate = startDate || today;
    const reportEndDate = endDate || today;

    const whereClause = {
      date: {
        [Op.between]: [reportStartDate, reportEndDate]
      }
    };

    if (dsoName) {
      whereClause.dsoName = {
        [Op.like]: `%${dsoName}%`
      };
    }

    const deliveries = await DSODelivery.findAll({
      where: whereClause,
      include: [{
        model: DSOAllocation,
        required: false
      }],
      order: [['date', 'DESC'], ['createdAt', 'DESC']]
    });

    const reportData = deliveries.map(delivery => {
      const allocations = delivery.dsoAllocations || [];
      const totalAllocations = allocations.reduce((acc, alloc) => {
        acc.cardsAllocated += alloc.cardsAllocated;
        acc.simsAllocated += alloc.simsAllocated;
        acc.kitsAllocated += alloc.kitsAllocated;
        acc.cardsSold += alloc.cardsSold;
        acc.simsSold += alloc.simsSold;
        acc.kitsSold += alloc.kitsSold;
        return acc;
      }, {
        cardsAllocated: 0,
        simsAllocated: 0,
        kitsAllocated: 0,
        cardsSold: 0,
        simsSold: 0,
        kitsSold: 0
      });

      return {
        date: delivery.date,
        dsoName: delivery.dsoName,
        amountPaid: delivery.amountPaid,
        cardsReceived: delivery.cardsReceivedTotal,
        cardTypes: delivery.cardTypes.map(ct => `${ct.type}: ${ct.quantity}`).join(', '),
        simsReceived: delivery.simsReceived,
        kitsReceived: delivery.kitsReceived,
        cardsAllocated: totalAllocations.cardsAllocated,
        simsAllocated: totalAllocations.simsAllocated,
        kitsAllocated: totalAllocations.kitsAllocated,
        cardsSold: totalAllocations.cardsSold,
        simsSold: totalAllocations.simsSold,
        kitsSold: totalAllocations.kitsSold,
        simSoldToday: totalAllocations.simsSold > 0 ? 'Yes' : 'No',
        kitSoldToday: totalAllocations.kitsSold > 0 ? 'Yes' : 'No',
        notes: delivery.notes,
        receivedBy: delivery.receivedBy
      };
    });

    // Calculate summary
    const summary = reportData.reduce((acc, item) => {
      acc.totalPaidToDSO += parseFloat(item.amountPaid);
      acc.totalCardsReceived += item.cardsReceived;
      acc.totalCardsAllocated += item.cardsAllocated;
      acc.totalCardsSold += item.cardsSold;
      return acc;
    }, {
      totalPaidToDSO: 0,
      totalCardsReceived: 0,
      totalCardsAllocated: 0,
      totalCardsSold: 0
    });

    res.send({
      reportData,
      summary,
      dateRange: {
        startDate: reportStartDate,
        endDate: reportEndDate
      }
    });
  } catch (error) {
    console.error('Error generating daily report:', error);
    res.status(500).send({
      message: "Error generating daily report",
      error: error.message
    });
  }
};

// Get DSO names for dropdown
exports.getDSONames = async (req, res) => {
  try {
    const dsoNames = await DSODelivery.findAll({
      attributes: ['dsoName'],
      group: ['dsoName'],
      order: [['dsoName', 'ASC']]
    });

    const names = dsoNames.map(item => item.dsoName);
    res.send(names);
  } catch (error) {
    console.error('Error getting DSO names:', error);
    res.status(500).send({
      message: "Error retrieving DSO names",
      error: error.message
    });
  }
};