const db = require("../models");
const Expense = db.expenses;
const Op = db.Sequelize.Op;

// Create and Save a new Expense
exports.create = async (req, res) => {
  try {
    // Validate request
    if (!req.body.description || !req.body.amount || !req.body.category) {
      return res.status(400).send({
        message: "Description, amount, and category are required!"
      });
    }

    // Create an Expense
    const expense = {
      description: req.body.description,
      amount: req.body.amount,
      category: req.body.category,
      paymentMethod: req.body.paymentMethod || 'cash',
      notes: req.body.notes,
      expenseDate: req.body.expenseDate || new Date()
    };

    // Save Expense in the database
    const data = await Expense.create(expense);
    res.send({
      message: "Expense created successfully!",
      expense: data
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the Expense."
    });
  }
};

// Retrieve all Expenses from the database with filtering
exports.findAll = async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 20, 
      category, 
      paymentMethod, 
      startDate, 
      endDate,
      search 
    } = req.query;

    const offset = (page - 1) * limit;
    let whereCondition = {};

    // Add filters
    if (category) {
      whereCondition.category = category;
    }

    if (paymentMethod) {
      whereCondition.paymentMethod = paymentMethod;
    }

    if (startDate && endDate) {
      whereCondition.expenseDate = {
        [Op.between]: [new Date(startDate), new Date(endDate + 'T23:59:59.999Z')]
      };
    } else if (startDate) {
      whereCondition.expenseDate = {
        [Op.gte]: new Date(startDate)
      };
    } else if (endDate) {
      whereCondition.expenseDate = {
        [Op.lte]: new Date(endDate + 'T23:59:59.999Z')
      };
    }

    if (search) {
      whereCondition[Op.or] = [
        { description: { [Op.like]: `%${search}%` } },
        { notes: { [Op.like]: `%${search}%` } }
      ];
    }

    const { count, rows } = await Expense.findAndCountAll({
      where: whereCondition,
      limit: parseInt(limit),
      offset: parseInt(offset),
      order: [['expenseDate', 'DESC'], ['createdAt', 'DESC']]
    });

    res.send({
      expenses: rows,
      totalItems: count,
      totalPages: Math.ceil(count / limit),
      currentPage: parseInt(page)
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving expenses."
    });
  }
};

// Find a single Expense with an id
exports.findOne = async (req, res) => {
  try {
    const id = req.params.id;
    const data = await Expense.findByPk(id);

    if (data) {
      res.send(data);
    } else {
      res.status(404).send({
        message: `Cannot find Expense with id=${id}.`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: "Error retrieving Expense with id=" + req.params.id
    });
  }
};

// Update an Expense by the id in the request
exports.update = async (req, res) => {
  try {
    const id = req.params.id;
    const [num] = await Expense.update(req.body, {
      where: { id: id }
    });

    if (num == 1) {
      const updatedExpense = await Expense.findByPk(id);
      res.send({
        message: "Expense was updated successfully.",
        expense: updatedExpense
      });
    } else {
      res.send({
        message: `Cannot update Expense with id=${id}. Maybe Expense was not found or req.body is empty!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: "Error updating Expense with id=" + req.params.id
    });
  }
};

// Delete an Expense with the specified id in the request
exports.delete = async (req, res) => {
  try {
    const id = req.params.id;
    const num = await Expense.destroy({
      where: { id: id }
    });

    if (num == 1) {
      res.send({
        message: "Expense was deleted successfully!"
      });
    } else {
      res.send({
        message: `Cannot delete Expense with id=${id}. Maybe Expense was not found!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: "Could not delete Expense with id=" + req.params.id
    });
  }
};

// Delete all Expenses from the database
exports.deleteAll = async (req, res) => {
  try {
    const nums = await Expense.destroy({
      where: {},
      truncate: false
    });
    res.send({ message: `${nums} Expenses were deleted successfully!` });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while removing all expenses."
    });
  }
};

// Get expense summary for dashboard
exports.getSummary = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    const today = new Date();
    const startOfToday = new Date(today.getFullYear(), today.getMonth(), today.getDate());
    const endOfToday = new Date(today.getFullYear(), today.getMonth(), today.getDate(), 23, 59, 59, 999);

    // Default to today if no dates provided
    const queryStartDate = startDate ? new Date(startDate) : startOfToday;
    const queryEndDate = endDate ? new Date(endDate + 'T23:59:59.999Z') : endOfToday;

    // Get total expenses for the period
    const totalExpenses = await Expense.sum('amount', {
      where: {
        expenseDate: {
          [Op.between]: [queryStartDate, queryEndDate]
        }
      }
    });

    // Get expenses by category
    const expensesByCategory = await Expense.findAll({
      attributes: [
        'category',
        [db.Sequelize.fn('SUM', db.Sequelize.col('amount')), 'total'],
        [db.Sequelize.fn('COUNT', db.Sequelize.col('id')), 'count']
      ],
      where: {
        expenseDate: {
          [Op.between]: [queryStartDate, queryEndDate]
        }
      },
      group: ['category'],
      order: [[db.Sequelize.fn('SUM', db.Sequelize.col('amount')), 'DESC']]
    });

    // Get expenses by payment method
    const expensesByPaymentMethod = await Expense.findAll({
      attributes: [
        'paymentMethod',
        [db.Sequelize.fn('SUM', db.Sequelize.col('amount')), 'total'],
        [db.Sequelize.fn('COUNT', db.Sequelize.col('id')), 'count']
      ],
      where: {
        expenseDate: {
          [Op.between]: [queryStartDate, queryEndDate]
        }
      },
      group: ['paymentMethod'],
      order: [[db.Sequelize.fn('SUM', db.Sequelize.col('amount')), 'DESC']]
    });

    // Get recent expenses
    const recentExpenses = await Expense.findAll({
      where: {
        expenseDate: {
          [Op.between]: [queryStartDate, queryEndDate]
        }
      },
      order: [['expenseDate', 'DESC'], ['createdAt', 'DESC']],
      limit: 5
    });

    // Get expense count
    const expenseCount = await Expense.count({
      where: {
        expenseDate: {
          [Op.between]: [queryStartDate, queryEndDate]
        }
      }
    });

    res.send({
      totalExpenses: totalExpenses || 0,
      expenseCount,
      expensesByCategory,
      expensesByPaymentMethod,
      recentExpenses,
      period: {
        startDate: queryStartDate,
        endDate: queryEndDate
      }
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving expense summary."
    });
  }
};

// Get daily expense summary
exports.getDailySummary = async (req, res) => {
  try {
    const { date } = req.query;
    const targetDate = date ? new Date(date) : new Date();
    const startOfDay = new Date(targetDate.getFullYear(), targetDate.getMonth(), targetDate.getDate());
    const endOfDay = new Date(targetDate.getFullYear(), targetDate.getMonth(), targetDate.getDate(), 23, 59, 59, 999);

    const summary = await exports.getSummary({
      query: {
        startDate: startOfDay.toISOString().split('T')[0],
        endDate: endOfDay.toISOString().split('T')[0]
      }
    }, res);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving daily expense summary."
    });
  }
};

// Get expense categories (for dropdown)
exports.getCategories = async (req, res) => {
  try {
    const categories = await Expense.findAll({
      attributes: [[db.Sequelize.fn('DISTINCT', db.Sequelize.col('category')), 'category']],
      order: [['category', 'ASC']]
    });

    const categoryList = categories.map(cat => cat.category);
    
    // Add some default categories if none exist
    const defaultCategories = [
      'Office Supplies',
      'Transportation',
      'Meals & Entertainment',
      'Utilities',
      'Rent',
      'Marketing',
      'Equipment',
      'Maintenance',
      'Insurance',
      'Professional Services',
      'Other'
    ];

    const allCategories = [...new Set([...categoryList, ...defaultCategories])].sort();

    res.send({
      categories: allCategories
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving categories."
    });
  }
};