const db = require('../models');
const Product = db.products;
const { Op } = require('sequelize');

// Create and Save a new Product
exports.create = async (req, res) => {
  try {
    // Validate request
    if (!req.body.name || !req.body.price) {
      return res.status(400).send({
        message: "Product name and price cannot be empty!"
      });
    }

    // Create a Product
    const product = {
      name: req.body.name,
      barcode: req.body.barcode,
      category: req.body.category,
      price: req.body.price,
      cost: req.body.cost || req.body.price, // Default cost to price if not provided
      stock: req.body.stock || 0,
      unit: req.body.unit,
      image: req.body.image,
      description: req.body.description
    };

    // Save Product in the database
    const data = await Product.create(product);
    res.status(201).send(data);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the Product."
    });
  }
};

// Retrieve all Products from the database
exports.findAll = async (req, res) => {
  try {
    const { name, category, barcode } = req.query;
    let condition = {};

    if (name) {
      condition.name = { [Op.like]: `%${name}%` };
    }
    if (category) {
      condition.category = category;
    }
    if (barcode) {
      condition.barcode = barcode;
    }

    const data = await Product.findAll({ where: condition });
    res.send(data);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving products."
    });
  }
};

// Find a single Product with an id
exports.findOne = async (req, res) => {
  try {
    const id = req.params.id;
    const data = await Product.findByPk(id);
    
    if (data) {
      res.send(data);
    } else {
      res.status(404).send({
        message: `Cannot find Product with id=${id}.`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Error retrieving Product with id=${req.params.id}`
    });
  }
};

// Find a Product by barcode
exports.findByBarcode = async (req, res) => {
  try {
    const barcode = req.params.barcode;
    const data = await Product.findOne({ where: { barcode: barcode } });
    
    if (data) {
      res.send(data);
    } else {
      res.status(404).send({
        message: `Cannot find Product with barcode=${barcode}.`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Error retrieving Product with barcode=${req.params.barcode}`
    });
  }
};

// Update a Product by the id in the request
exports.update = async (req, res) => {
  try {
    const id = req.params.id;
    const [num] = await Product.update(req.body, {
      where: { id: id }
    });

    if (num === 1) {
      res.send({
        message: "Product was updated successfully."
      });
    } else {
      res.send({
        message: `Cannot update Product with id=${id}. Maybe Product was not found or req.body is empty!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Error updating Product with id=${req.params.id}`
    });
  }
};

// Update stock quantity
exports.updateStock = async (req, res) => {
  try {
    const id = req.params.id;
    const { quantity, operation } = req.body;
    
    if (!quantity || !operation) {
      return res.status(400).send({
        message: "Quantity and operation type are required"
      });
    }

    const product = await Product.findByPk(id);
    if (!product) {
      return res.status(404).send({
        message: `Product with id=${id} not found`
      });
    }

    let newStock = product.stock;
    if (operation === 'add') {
      newStock += parseInt(quantity);
    } else if (operation === 'subtract') {
      newStock -= parseInt(quantity);
      if (newStock < 0) {
        return res.status(400).send({
          message: "Stock cannot be negative"
        });
      }
    } else {
      return res.status(400).send({
        message: "Invalid operation. Use 'add' or 'subtract'"
      });
    }

    await Product.update({ stock: newStock }, {
      where: { id: id }
    });

    res.send({
      message: "Stock updated successfully",
      newStock: newStock
    });
  } catch (err) {
    res.status(500).send({
      message: `Error updating stock for Product with id=${req.params.id}`
    });
  }
};

// Delete a Product with the specified id
exports.delete = async (req, res) => {
  try {
    const id = req.params.id;
    const num = await Product.destroy({
      where: { id: id }
    });

    if (num === 1) {
      res.send({
        message: "Product was deleted successfully!"
      });
    } else {
      res.send({
        message: `Cannot delete Product with id=${id}. Maybe Product was not found!`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Could not delete Product with id=${req.params.id}`
    });
  }
};

// Delete all Products from the database
exports.deleteAll = async (req, res) => {
  try {
    const nums = await Product.destroy({
      where: {},
      truncate: false
    });
    
    res.send({
      message: `${nums} Products were deleted successfully!`
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while removing all products."
    });
  }
};