const db = require('../models');
const RechargeTransaction = db.rechargeTransactions;
const { Op } = require('sequelize');

// Create a new recharge transaction
exports.create = async (req, res) => {
  try {
    const {
      customer_number,
      transaction_id,
      amount,
      old_balance,
      new_balance,
      commission,
      transaction_type,
      ocr_confidence,
      is_verified,
      ocr_raw_text,
      notes
    } = req.body;

    // Validate required fields
    if (!customer_number || !transaction_id || !amount || old_balance === undefined || new_balance === undefined) {
      return res.status(400).json({
        success: false,
        message: 'Missing required fields: customer_number, transaction_id, amount, old_balance, new_balance'
      });
    }

    // Check if transaction ID already exists
    const existingTransaction = await RechargeTransaction.findOne({
      where: { transaction_id }
    });

    if (existingTransaction) {
      return res.status(409).json({
        success: false,
        message: 'Transaction ID already exists',
        data: existingTransaction
      });
    }

    // Create the transaction
    const transaction = await RechargeTransaction.create({
      customer_number,
      transaction_id,
      amount: parseFloat(amount),
      old_balance: parseFloat(old_balance),
      new_balance: parseFloat(new_balance),
      commission: commission ? parseFloat(commission) : 0.00,
      transaction_type: transaction_type || 'mobile_recharge',
      ocr_confidence: ocr_confidence ? parseFloat(ocr_confidence) : null,
      is_verified: is_verified || false,
      ocr_raw_text,
      notes
    });

    res.status(201).json({
      success: true,
      message: 'Recharge transaction created successfully',
      data: transaction
    });

  } catch (error) {
    console.error('Error creating recharge transaction:', error);
    res.status(500).json({
      success: false,
      message: 'Error creating recharge transaction',
      error: error.message
    });
  }
};

// Get all recharge transactions with pagination and filters
exports.getAll = async (req, res) => {
  try {
    const {
      page = 1,
      limit = 50,
      transaction_type,
      customer_number,
      start_date,
      end_date,
      is_verified,
      sort_by = 'created_at',
      sort_order = 'DESC'
    } = req.query;

    const offset = (page - 1) * limit;
    const whereClause = {};

    // Apply filters
    if (transaction_type) {
      whereClause.transaction_type = transaction_type;
    }

    if (customer_number) {
      whereClause.customer_number = {
        [Op.like]: `%${customer_number}%`
      };
    }

    if (is_verified !== undefined) {
      whereClause.is_verified = is_verified === 'true';
    }

    if (start_date && end_date) {
      whereClause.created_at = {
        [Op.between]: [new Date(start_date), new Date(end_date)]
      };
    } else if (start_date) {
      whereClause.created_at = {
        [Op.gte]: new Date(start_date)
      };
    } else if (end_date) {
      whereClause.created_at = {
        [Op.lte]: new Date(end_date)
      };
    }

    const { count, rows } = await RechargeTransaction.findAndCountAll({
      where: whereClause,
      limit: parseInt(limit),
      offset: parseInt(offset),
      order: [[sort_by, sort_order.toUpperCase()]]
    });

    res.json({
      success: true,
      data: {
        transactions: rows,
        pagination: {
          total: count,
          page: parseInt(page),
          limit: parseInt(limit),
          totalPages: Math.ceil(count / limit)
        }
      }
    });

  } catch (error) {
    console.error('Error fetching recharge transactions:', error);
    res.status(500).json({
      success: false,
      message: 'Error fetching recharge transactions',
      error: error.message
    });
  }
};

// Get a single recharge transaction by ID
exports.getById = async (req, res) => {
  try {
    const { id } = req.params;

    const transaction = await RechargeTransaction.findByPk(id);

    if (!transaction) {
      return res.status(404).json({
        success: false,
        message: 'Recharge transaction not found'
      });
    }

    res.json({
      success: true,
      data: transaction
    });

  } catch (error) {
    console.error('Error fetching recharge transaction:', error);
    res.status(500).json({
      success: false,
      message: 'Error fetching recharge transaction',
      error: error.message
    });
  }
};

// Update a recharge transaction
exports.update = async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;

    const transaction = await RechargeTransaction.findByPk(id);

    if (!transaction) {
      return res.status(404).json({
        success: false,
        message: 'Recharge transaction not found'
      });
    }

    // Convert numeric fields
    if (updateData.amount) updateData.amount = parseFloat(updateData.amount);
    if (updateData.old_balance) updateData.old_balance = parseFloat(updateData.old_balance);
    if (updateData.new_balance) updateData.new_balance = parseFloat(updateData.new_balance);
    if (updateData.commission) updateData.commission = parseFloat(updateData.commission);
    if (updateData.ocr_confidence) updateData.ocr_confidence = parseFloat(updateData.ocr_confidence);

    await transaction.update(updateData);

    res.json({
      success: true,
      message: 'Recharge transaction updated successfully',
      data: transaction
    });

  } catch (error) {
    console.error('Error updating recharge transaction:', error);
    res.status(500).json({
      success: false,
      message: 'Error updating recharge transaction',
      error: error.message
    });
  }
};

// Delete a recharge transaction
exports.delete = async (req, res) => {
  try {
    const { id } = req.params;

    const transaction = await RechargeTransaction.findByPk(id);

    if (!transaction) {
      return res.status(404).json({
        success: false,
        message: 'Recharge transaction not found'
      });
    }

    await transaction.destroy();

    res.json({
      success: true,
      message: 'Recharge transaction deleted successfully'
    });

  } catch (error) {
    console.error('Error deleting recharge transaction:', error);
    res.status(500).json({
      success: false,
      message: 'Error deleting recharge transaction',
      error: error.message
    });
  }
};

// Get daily summary report
exports.getDailySummary = async (req, res) => {
  try {
    const { date } = req.query;
    const targetDate = date ? new Date(date) : new Date();
    
    // Set date range for the day
    const startOfDay = new Date(targetDate);
    startOfDay.setHours(0, 0, 0, 0);
    
    const endOfDay = new Date(targetDate);
    endOfDay.setHours(23, 59, 59, 999);

    // Get transactions for the day
    const transactions = await RechargeTransaction.findAll({
      where: {
        created_at: {
          [Op.between]: [startOfDay, endOfDay]
        }
      },
      order: [['created_at', 'ASC']]
    });

    // Calculate summary
    const summary = {
      date: targetDate.toISOString().split('T')[0],
      total_transactions: transactions.length,
      total_amount: 0,
      total_commission: 0,
      opening_balance: null,
      closing_balance: null,
      transaction_types: {
        cash_in: { count: 0, amount: 0, commission: 0 },
        cash_out: { count: 0, amount: 0, commission: 0 },
        mobile_recharge: { count: 0, amount: 0, commission: 0 }
      },
      verified_transactions: 0,
      unverified_transactions: 0
    };

    transactions.forEach((transaction, index) => {
      // Set opening balance from first transaction
      if (index === 0) {
        summary.opening_balance = transaction.old_balance;
      }
      
      // Set closing balance from last transaction
      if (index === transactions.length - 1) {
        summary.closing_balance = transaction.new_balance;
      }

      // Add to totals
      summary.total_amount += parseFloat(transaction.amount);
      summary.total_commission += parseFloat(transaction.commission);

      // Add to transaction type summary
      const type = transaction.transaction_type;
      if (summary.transaction_types[type]) {
        summary.transaction_types[type].count++;
        summary.transaction_types[type].amount += parseFloat(transaction.amount);
        summary.transaction_types[type].commission += parseFloat(transaction.commission);
      }

      // Count verified/unverified
      if (transaction.is_verified) {
        summary.verified_transactions++;
      } else {
        summary.unverified_transactions++;
      }
    });

    res.json({
      success: true,
      data: {
        summary,
        transactions
      }
    });

  } catch (error) {
    console.error('Error generating daily summary:', error);
    res.status(500).json({
      success: false,
      message: 'Error generating daily summary',
      error: error.message
    });
  }
};

// Verify a transaction
exports.verify = async (req, res) => {
  try {
    const { id } = req.params;
    const { is_verified, notes } = req.body;

    const transaction = await RechargeTransaction.findByPk(id);

    if (!transaction) {
      return res.status(404).json({
        success: false,
        message: 'Recharge transaction not found'
      });
    }

    await transaction.update({
      is_verified: is_verified !== undefined ? is_verified : true,
      notes: notes || transaction.notes
    });

    res.json({
      success: true,
      message: 'Transaction verification updated successfully',
      data: transaction
    });

  } catch (error) {
    console.error('Error verifying transaction:', error);
    res.status(500).json({
      success: false,
      message: 'Error verifying transaction',
      error: error.message
    });
  }
};