const db = require('../models');
const { Op } = require('sequelize');
const Product = db.products;
const Sale = db.sales;
const SaleItem = db.saleItems;
const User = db.users;
const Agent = db.agents;
const Payment = db.payments;

// Get sales summary by date range
exports.getSalesSummary = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    const whereClause = {};
    if (startDate && endDate) {
      whereClause.createdAt = {
        [Op.between]: [new Date(startDate), new Date(endDate)]
      };
    }
    
    const salesData = await Sale.findAll({
      where: whereClause,
      attributes: [
        [db.sequelize.fn('date', db.sequelize.col('createdAt')), 'date'],
        [db.sequelize.fn('count', db.sequelize.col('id')), 'count'],
        [db.sequelize.fn('sum', db.sequelize.col('totalAmount')), 'totalSales'],
        [db.sequelize.fn('sum', db.sequelize.col('discount')), 'totalDiscount'],
        [db.sequelize.fn('sum', db.sequelize.col('tax')), 'totalTax'],
      ],
      group: [db.sequelize.fn('date', db.sequelize.col('createdAt'))],
      order: [[db.sequelize.fn('date', db.sequelize.col('createdAt')), 'ASC']]
    });
    
    const totalSales = await Sale.count({
      where: whereClause
    });
    
    const totalRevenue = await Sale.sum('totalAmount', {
      where: whereClause
    });
    
    const averageTicket = totalRevenue / totalSales || 0;
    
    return res.status(200).json({
      success: true,
      data: {
        salesByDate: salesData,
        summary: {
          totalSales,
          totalRevenue,
          averageTicket
        }
      }
    });
  } catch (error) {
    console.error('Error getting sales summary:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to get sales summary',
      error: error.message
    });
  }
};

// Get top selling products
exports.getTopProducts = async (req, res) => {
  try {
    const { limit = 10, startDate, endDate } = req.query;
    
    const whereClause = {};
    if (startDate && endDate) {
      whereClause.createdAt = {
        [Op.between]: [new Date(startDate), new Date(endDate)]
      };
    }
    
    const topProducts = await SaleItem.findAll({
      attributes: [
        'productId',
        [db.sequelize.fn('sum', db.sequelize.col('quantity')), 'totalQuantity'],
        [db.sequelize.fn('sum', db.sequelize.col('subtotal')), 'totalRevenue']
      ],
      include: [
        {
          model: Product,
          attributes: ['name', 'sku', 'category']
        },
        {
          model: Sale,
          attributes: [],
          where: whereClause
        }
      ],
      group: ['productId', 'product.id'],
      order: [[db.sequelize.fn('sum', db.sequelize.col('quantity')), 'DESC']],
      limit: parseInt(limit)
    });
    
    return res.status(200).json({
      success: true,
      data: topProducts
    });
  } catch (error) {
    console.error('Error getting top products:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to get top products',
      error: error.message
    });
  }
};

// Get inventory status
exports.getInventoryStatus = async (req, res) => {
  try {
    const lowStockThreshold = req.query.threshold || 10;
    
    const lowStockProducts = await Product.findAll({
      where: {
        stock: { [Op.lt]: lowStockThreshold },
        status: 'active'
      },
      order: [['stock', 'ASC']]
    });
    
    const outOfStockCount = await Product.count({
      where: {
        stock: 0,
        status: 'active'
      }
    });
    
    const totalProducts = await Product.count({
      where: { status: 'active' }
    });
    
    const totalInventoryValue = await Product.sum(
      db.sequelize.literal('stock * purchasePrice')
    );
    
    return res.status(200).json({
      success: true,
      data: {
        lowStockProducts,
        summary: {
          outOfStockCount,
          lowStockCount: lowStockProducts.length,
          totalProducts,
          totalInventoryValue
        }
      }
    });
  } catch (error) {
    console.error('Error getting inventory status:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to get inventory status',
      error: error.message
    });
  }
};

// Get user performance
exports.getUserPerformance = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    const whereClause = {};
    if (startDate && endDate) {
      whereClause.createdAt = {
        [Op.between]: [new Date(startDate), new Date(endDate)]
      };
    }
    
    const userPerformance = await Sale.findAll({
      attributes: [
        'userId',
        [db.sequelize.fn('count', db.sequelize.col('id')), 'saleCount'],
        [db.sequelize.fn('sum', db.sequelize.col('totalAmount')), 'totalRevenue']
      ],
      include: [
        {
          model: User,
          attributes: ['username', 'fullName']
        }
      ],
      where: whereClause,
      group: ['userId', 'user.id'],
      order: [[db.sequelize.fn('sum', db.sequelize.col('totalAmount')), 'DESC']]
    });
    
    return res.status(200).json({
      success: true,
      data: userPerformance
    });
  } catch (error) {
    console.error('Error getting user performance:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to get user performance',
      error: error.message
    });
  }
};

// Get agent balances
exports.getAgentBalances = async (req, res) => {
  try {
    const agents = await Agent.findAll({
      attributes: [
        'id', 
        'name', 
        'company', 
        'phone', 
        'email', 
        'balance',
        'status'
      ],
      order: [['balance', 'DESC']]
    });
    
    const totalDue = await Agent.sum('balance', {
      where: {
        balance: { [Op.gt]: 0 }
      }
    });
    
    const totalCredit = await Agent.sum('balance', {
      where: {
        balance: { [Op.lt]: 0 }
      }
    });
    
    return res.status(200).json({
      success: true,
      data: {
        agents,
        summary: {
          totalDue: totalDue || 0,
          totalCredit: Math.abs(totalCredit) || 0,
          netBalance: (totalDue || 0) - Math.abs(totalCredit || 0)
        }
      }
    });
  } catch (error) {
    console.error('Error getting agent balances:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to get agent balances',
      error: error.message
    });
  }
};

// Get payment summary
exports.getPaymentSummary = async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    
    const whereClause = {};
    if (startDate && endDate) {
      whereClause.createdAt = {
        [Op.between]: [new Date(startDate), new Date(endDate)]
      };
    }
    
    const paymentsByMethod = await Payment.findAll({
      attributes: [
        'method',
        [db.sequelize.fn('count', db.sequelize.col('id')), 'count'],
        [db.sequelize.fn('sum', db.sequelize.col('amount')), 'total']
      ],
      where: whereClause,
      group: ['method'],
      order: [[db.sequelize.fn('sum', db.sequelize.col('amount')), 'DESC']]
    });
    
    const paymentsByType = await Payment.findAll({
      attributes: [
        'type',
        [db.sequelize.fn('count', db.sequelize.col('id')), 'count'],
        [db.sequelize.fn('sum', db.sequelize.col('amount')), 'total']
      ],
      where: whereClause,
      group: ['type'],
      order: [['type', 'ASC']]
    });
    
    return res.status(200).json({
      success: true,
      data: {
        paymentsByMethod,
        paymentsByType
      }
    });
  } catch (error) {
    console.error('Error getting payment summary:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to get payment summary',
      error: error.message
    });
  }
};