const db = require('../models');
const Sale = db.sales;
const SaleItem = db.saleItems;
const Product = db.products;
const Customer = db.customers;
const { Op } = require('sequelize');

// Create and Save a new Sale
exports.create = async (req, res) => {
  try {
    // Validate request
    if (!req.body.items || !req.body.items.length) {
      return res.status(400).send({
        message: "Sale must have at least one item!"
      });
    }

    // Start a transaction
    const t = await db.sequelize.transaction();

    try {
      // Create a Sale
      const sale = {
        totalAmount: req.body.grandTotal || req.body.totalAmount,
        discount: req.body.discount || 0,
        tax: req.body.tax || 0,
        paymentMethod: req.body.paymentMethod,
        paymentDetails: req.body.paymentDetails,
        customerId: req.body.customerId,
        userId: req.body.userId,
        status: 'completed'
      };

      // Save Sale in the database
      const newSale = await Sale.create(sale, { transaction: t });

      // Create sale items and update product stock
      const saleItems = [];
      for (const item of req.body.items) {
        // Create sale item
        const saleItem = await SaleItem.create({
          saleId: newSale.id,
          productId: item.productId,
          quantity: item.quantity,
          price: item.price,
          total: item.total
        }, { transaction: t });
        
        saleItems.push(saleItem);

        // Update product stock
        const product = await Product.findByPk(item.productId, { transaction: t });
        if (!product) {
          throw new Error(`Product with id=${item.productId} not found`);
        }
        
        const newStock = product.stock - item.quantity;
        if (newStock < 0) {
          throw new Error(`Insufficient stock for product: ${product.name}`);
        }
        
        await Product.update({ stock: newStock }, {
          where: { id: item.productId },
          transaction: t
        });
      }

      // If customer exists, update their purchase history
      if (req.body.customerId) {
        const customer = await Customer.findByPk(req.body.customerId, { transaction: t });
        if (customer) {
          await Customer.update(
            { 
              totalPurchases: customer.totalPurchases + 1,
              lastPurchaseDate: new Date()
            },
            { 
              where: { id: req.body.customerId },
              transaction: t
            }
          );
        }
      }

      // Commit the transaction
      await t.commit();

      res.status(201).send({
        id: newSale.id,
        ...newSale.dataValues,
        items: saleItems
      });
    } catch (error) {
      // Rollback the transaction in case of error
      await t.rollback();
      throw error;
    }
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while creating the Sale."
    });
  }
};

// Retrieve all Sales from the database
exports.findAll = async (req, res) => {
  try {
    const { startDate, endDate, paymentMethod, userId, customerId, limit } = req.query;
    let condition = {};
    let queryOptions = {
      where: condition,
      include: [
        {
          model: SaleItem,
          include: [{
            model: Product
          }]
        },
        {
          model: Customer
        }
      ],
      order: [['createdAt', 'DESC']]
    };

    if (startDate && endDate) {
      condition.createdAt = { 
        [Op.between]: [new Date(startDate), new Date(endDate)]
      };
    } else if (startDate) {
      condition.createdAt = { [Op.gte]: new Date(startDate) };
    } else if (endDate) {
      condition.createdAt = { [Op.lte]: new Date(endDate) };
    }

    if (paymentMethod) {
      condition.paymentMethod = paymentMethod;
    }
    
    if (userId) {
      condition.userId = userId;
    }
    
    if (customerId) {
      condition.customerId = customerId;
    }

    if (limit) {
      queryOptions.limit = parseInt(limit);
    }

    const data = await Sale.findAll(queryOptions);
    
    res.send(data);
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving sales."
    });
  }
};

// Find a single Sale with an id
exports.findOne = async (req, res) => {
  try {
    const id = req.params.id;
    const data = await Sale.findByPk(id, {
      include: [
        {
          model: SaleItem,
          include: [{
            model: Product
          }]
        },
        {
          model: Customer
        }
      ]
    });
    
    if (data) {
      res.send(data);
    } else {
      res.status(404).send({
        message: `Cannot find Sale with id=${id}.`
      });
    }
  } catch (err) {
    res.status(500).send({
      message: `Error retrieving Sale with id=${req.params.id}`
    });
  }
};

// Get sales summary (for dashboard)
exports.getSummary = async (req, res) => {
  try {
    const { period } = req.query;
    let startDate, endDate = new Date();
    
    // Set time period
    switch(period) {
      case 'today':
        startDate = new Date();
        startDate.setHours(0, 0, 0, 0);
        break;
      case 'yesterday':
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 1);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(startDate);
        endDate.setHours(23, 59, 59, 999);
        break;
      case 'week':
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 7);
        break;
      case 'month':
        startDate = new Date();
        startDate.setMonth(startDate.getMonth() - 1);
        break;
      default:
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 30); // Default to last 30 days
    }

    // Get total sales
    const totalSales = await Sale.count({
      where: {
        createdAt: { 
          [Op.between]: [startDate, endDate]
        }
      }
    });

    // Get total revenue
    const revenue = await Sale.sum('totalAmount', {
      where: {
        createdAt: { 
          [Op.between]: [startDate, endDate]
        }
      }
    });

    // Get payment method breakdown
    const paymentMethods = await Sale.findAll({
      attributes: [
        'paymentMethod',
        [db.sequelize.fn('COUNT', db.sequelize.col('id')), 'count'],
        [db.sequelize.fn('SUM', db.sequelize.col('totalAmount')), 'total']
      ],
      where: {
        createdAt: { 
          [Op.between]: [startDate, endDate]
        }
      },
      group: ['paymentMethod']
    });

    const topProducts = await SaleItem.findAll({
      attributes: [
        'productId',
        [db.sequelize.fn('SUM', db.sequelize.col('quantity')), 'totalQuantity'],
        [db.sequelize.fn('SUM', db.sequelize.col('totalPrice')), 'totalRevenue']
      ],
      include: [{
        model: Product,
        attributes: ['name', 'price']
      }],
      group: ['productId'],
      order: [[db.sequelize.fn('SUM', db.sequelize.col('quantity')), 'DESC']],
      limit: 5
    });

    res.send({
      period,
      totalSales: totalSales || 0,
      revenue: revenue || 0,
      paymentMethods,
      topProducts
    });
  } catch (err) {
    res.status(500).send({
      message: err.message || "Some error occurred while retrieving sales summary."
    });
  }
};