const db = require('../models');
const MobileRechargeTransaction = db.mobileRechargeTransactions;
const MobileRechargeAccount = db.mobileRechargeAccounts;
const { Op } = require('sequelize');

// SMS patterns for different operators
const SMS_PATTERNS = {
  grameenphone: {
    patterns: [
      /(?:grameenphone|gp|grameen)\s*(?:recharge|রিচার্জ)\s*(?:of|এর)?\s*(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:to|এ|এর জন্য)?\s*(\d{11})/i,
      /(\d{11})\s*(?:recharged|রিচার্জ)\s*(?:with|দিয়ে)?\s*(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:from|থেকে)?\s*(?:grameenphone|gp|grameen)/i,
      /(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:recharge|রিচার্জ)\s*(?:to|এ|এর জন্য)?\s*(\d{11})\s*(?:grameenphone|gp|grameen)/i
    ],
    commissionRate: 2.80
  },
  banglalink: {
    patterns: [
      /(?:banglalink|bl)\s*(?:recharge|রিচার্জ)\s*(?:of|এর)?\s*(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:to|এ|এর জন্য)?\s*(\d{11})/i,
      /(\d{11})\s*(?:recharged|রিচার্জ)\s*(?:with|দিয়ে)?\s*(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:from|থেকে)?\s*(?:banglalink|bl)/i,
      /(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:recharge|রিচার্জ)\s*(?:to|এ|এর জন্য)?\s*(\d{11})\s*(?:banglalink|bl)/i
    ],
    commissionRate: 2.50
  },
  robi: {
    patterns: [
      /(?:robi|airtel)\s*(?:recharge|রিচার্জ)\s*(?:of|এর)?\s*(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:to|এ|এর জন্য)?\s*(\d{11})/i,
      /(\d{11})\s*(?:recharged|রিচার্জ)\s*(?:with|দিয়ে)?\s*(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:from|থেকে)?\s*(?:robi|airtel)/i,
      /(?:tk|৳|taka)?\s*(\d+(?:\.\d{2})?)\s*(?:recharge|রিচার্জ)\s*(?:to|এ|এর জন্য)?\s*(\d{11})\s*(?:robi|airtel)/i
    ],
    commissionRate: 2.30
  }
};

// Function to detect operator from phone number
const detectOperatorFromNumber = (phoneNumber) => {
  const number = phoneNumber.replace(/^\+?88/, ''); // Remove country code
  
  // Grameenphone prefixes
  if (/^(017|013|019|014)/.test(number)) {
    return 'grameenphone';
  }
  
  // Banglalink prefixes
  if (/^(019|014|018)/.test(number)) {
    return 'banglalink';
  }
  
  // Robi/Airtel prefixes
  if (/^(018|016|017)/.test(number)) {
    return 'robi';
  }
  
  return null;
};

// Function to parse SMS content
const parseSMSContent = (smsText) => {
  const cleanText = smsText.trim();
  
  for (const [operator, config] of Object.entries(SMS_PATTERNS)) {
    for (const pattern of config.patterns) {
      const match = cleanText.match(pattern);
      if (match) {
        let amount, phoneNumber;
        
        // Different patterns may have amount and phone number in different positions
        if (match[1] && match[2]) {
          // Check which one is the phone number (11 digits)
          if (match[1].length === 11) {
            phoneNumber = match[1];
            amount = parseFloat(match[2]);
          } else if (match[2].length === 11) {
            phoneNumber = match[2];
            amount = parseFloat(match[1]);
          }
        }
        
        if (phoneNumber && amount && phoneNumber.length === 11) {
          // Verify operator from phone number
          const detectedOperator = detectOperatorFromNumber(phoneNumber);
          const finalOperator = detectedOperator || operator;
          
          return {
            operator: finalOperator,
            phoneNumber: phoneNumber,
            amount: amount,
            commissionRate: SMS_PATTERNS[finalOperator]?.commissionRate || 2.80,
            commission: (amount * (SMS_PATTERNS[finalOperator]?.commissionRate || 2.80)) / 100
          };
        }
      }
    }
  }
  
  return null;
};

// Receive SMS and process recharge
exports.receiveSMS = async (req, res) => {
  try {
    const { 
      smsText, 
      senderNumber, 
      receivedAt,
      messageId 
    } = req.body;

    if (!smsText) {
      return res.status(400).json({
        success: false,
        message: "SMS text is required"
      });
    }

    // Parse SMS content
    const parsedData = parseSMSContent(smsText);
    
    if (!parsedData) {
      return res.status(400).json({
        success: false,
        message: "Could not parse recharge information from SMS",
        smsText: smsText
      });
    }

    const { operator, phoneNumber, amount, commissionRate, commission } = parsedData;
    
    // Get current date and time
    const now = new Date();
    const date = now.toISOString().split('T')[0];
    const time = now.toTimeString().split(' ')[0].substring(0, 5);

    // Get or create today's account for the operator
    let account = await MobileRechargeAccount.findOne({
      where: { 
        date: date,
        operator: operator
      }
    });

    if (!account) {
      // Create today's account if it doesn't exist
      account = await MobileRechargeAccount.create({
        date: date,
        operator: operator,
        openingBalance: 0,
        totalRecharge: 0,
        totalCommission: 0,
        commissionRate: commissionRate
      });
    }

    // Create the recharge transaction
    const transaction = await MobileRechargeTransaction.create({
      date: date,
      time: time,
      operator: operator,
      customerNumber: phoneNumber,
      rechargeAmount: amount,
      commission: commission,
      srPay: 0,
      minutesCard: 0,
      mbCard: 0,
      simQuantity: 0,
      transactionType: 'recharge',
      reference: messageId || `SMS-${Date.now()}`,
      notes: `Auto-created from SMS: ${smsText.substring(0, 100)}...`
    });

    // Update account totals
    const newTotalRecharge = parseFloat(account.totalRecharge) + amount;
    const newTotalCommission = parseFloat(account.totalCommission) + commission;
    const openingBalance = parseFloat(account.openingBalance) || 0;
    
    // Calculate new closing balance
    const calculatedClosingBalance = openingBalance + newTotalRecharge + newTotalCommission;
    const calculatedNetChange = calculatedClosingBalance - openingBalance;

    await account.update({
      totalRecharge: newTotalRecharge,
      totalCommission: newTotalCommission,
      closingBalance: calculatedClosingBalance,
      netChange: calculatedNetChange
    });

    res.json({
      success: true,
      message: "Recharge transaction created successfully from SMS",
      data: {
        transaction: transaction,
        parsedData: parsedData,
        account: {
          operator: operator,
          date: date,
          totalRecharge: newTotalRecharge,
          totalCommission: newTotalCommission,
          closingBalance: calculatedClosingBalance
        }
      }
    });

  } catch (error) {
    console.error('Error processing SMS recharge:', error);
    res.status(500).json({
      success: false,
      message: "Error processing SMS recharge",
      error: error.message
    });
  }
};

// Get SMS processing history
exports.getSMSHistory = async (req, res) => {
  try {
    const { date, operator, limit = 50 } = req.query;
    const queryDate = date || new Date().toISOString().split('T')[0];

    const whereClause = {
      date: queryDate,
      reference: {
        [Op.like]: 'SMS-%'
      }
    };

    if (operator) {
      whereClause.operator = operator;
    }

    const transactions = await MobileRechargeTransaction.findAll({
      where: whereClause,
      order: [['time', 'DESC']],
      limit: parseInt(limit)
    });

    res.json({
      success: true,
      data: transactions,
      count: transactions.length
    });

  } catch (error) {
    console.error('Error getting SMS history:', error);
    res.status(500).json({
      success: false,
      message: "Error retrieving SMS history",
      error: error.message
    });
  }
};

// Test SMS parsing (for development/testing)
exports.testSMSParsing = async (req, res) => {
  try {
    const { smsText } = req.body;

    if (!smsText) {
      return res.status(400).json({
        success: false,
        message: "SMS text is required for testing"
      });
    }

    const parsedData = parseSMSContent(smsText);

    res.json({
      success: true,
      smsText: smsText,
      parsedData: parsedData,
      canProcess: parsedData !== null
    });

  } catch (error) {
    console.error('Error testing SMS parsing:', error);
    res.status(500).json({
      success: false,
      message: "Error testing SMS parsing",
      error: error.message
    });
  }
};