module.exports = (sequelize, Sequelize) => {
  const DSODelivery = sequelize.define("dsoDelivery", {
    id: {
      type: Sequelize.INTEGER,
      primaryKey: true,
      autoIncrement: true
    },
    date: {
      type: Sequelize.DATEONLY,
      allowNull: false,
      defaultValue: Sequelize.literal('CURRENT_DATE'),
      comment: "Delivery date"
    },
    dsoName: {
      type: Sequelize.STRING(100),
      allowNull: false,
      comment: "Distribution Sales Officer name"
    },
    receivedBy: {
      type: Sequelize.STRING(100),
      allowNull: false,
      comment: "User who received the delivery"
    },
    amountPaid: {
      type: Sequelize.DECIMAL(12, 2),
      allowNull: false,
      defaultValue: 0.00,
      validate: {
        min: 0
      },
      comment: "Amount paid to DSO in BDT"
    },
    cardsReceivedTotal: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Total number of cards received"
    },
    cardTypes: {
      type: Sequelize.JSON,
      allowNull: true,
      defaultValue: [],
      comment: "Array of card types with quantities: [{type: string, quantity: number}]"
    },
    simsReceived: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of SIM cards received"
    },
    kitsReceived: {
      type: Sequelize.INTEGER,
      allowNull: false,
      defaultValue: 0,
      validate: {
        min: 0
      },
      comment: "Number of kits received"
    },
    notes: {
      type: Sequelize.TEXT,
      allowNull: true,
      comment: "Additional notes about the delivery"
    },
    createdBy: {
      type: Sequelize.STRING(100),
      allowNull: false,
      comment: "User who created this record"
    },
    editedBy: {
      type: Sequelize.STRING(100),
      allowNull: true,
      comment: "User who last edited this record"
    },
    status: {
      type: Sequelize.ENUM('active', 'cancelled'),
      allowNull: false,
      defaultValue: 'active',
      comment: "Delivery status"
    }
  }, {
    timestamps: true,
    createdAt: 'createdAt',
    updatedAt: 'editedAt',
    hooks: {
      beforeValidate: (delivery, options) => {
        // Validate that at least one item type is received
        if (delivery.cardsReceivedTotal === 0 && delivery.simsReceived === 0 && delivery.kitsReceived === 0) {
          throw new Error('At least one of Cards, SIMs, or Kits must be received (quantity > 0)');
        }
        
        // Validate card types total matches cardsReceivedTotal (only if card types are filled)
        if (delivery.cardTypes && delivery.cardTypes.length > 0) {
          // Filter out empty card type entries
          const filledCardTypes = delivery.cardTypes.filter(ct => 
            (ct.type && ct.type.trim() !== '') || (ct.quantity && ct.quantity > 0)
          );
          
          // Only validate if user has actually filled in card type information
          if (filledCardTypes.length > 0) {
            const totalFromTypes = filledCardTypes.reduce((sum, cardType) => sum + (cardType.quantity || 0), 0);
            if (totalFromTypes !== delivery.cardsReceivedTotal) {
              throw new Error('Sum of card type quantities must equal total cards received');
            }
          }
        }
      }
    }
  });

  return DSODelivery;
};