const express = require('express');
const router = express.Router();
const db = require('../models');
const Product = db.products;
const Category = db.categories;

// Get inventory overview
router.get('/', async (req, res) => {
  try {
    const products = await Product.findAll({
      include: [{
        model: Category,
        attributes: ['name']
      }],
      order: [['name', 'ASC']]
    });

    const inventory = products.map(product => ({
      id: product.id,
      name: product.name,
      category: product.category ? product.category.name : 'Uncategorized',
      stock: product.stock,
      price: product.price,
      status: product.stock <= 5 ? 'Low Stock' : product.stock <= 10 ? 'Medium Stock' : 'In Stock'
    }));

    res.json(inventory);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
});

// Get low stock items
router.get('/low-stock', async (req, res) => {
  try {
    const lowStockProducts = await Product.findAll({
      where: {
        stock: {
          [db.Sequelize.Op.lte]: 10
        }
      },
      include: [{
        model: Category,
        attributes: ['name']
      }],
      order: [['stock', 'ASC']]
    });

    res.json(lowStockProducts);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
});

// Update stock
router.put('/stock/:id', async (req, res) => {
  try {
    const { stock, operation } = req.body; // operation: 'add', 'subtract', 'set'
    const product = await Product.findByPk(req.params.id);
    
    if (!product) {
      return res.status(404).json({ message: 'Product not found' });
    }

    let newStock;
    switch (operation) {
      case 'add':
        newStock = product.stock + parseInt(stock);
        break;
      case 'subtract':
        newStock = Math.max(0, product.stock - parseInt(stock));
        break;
      case 'set':
        newStock = parseInt(stock);
        break;
      default:
        return res.status(400).json({ message: 'Invalid operation' });
    }

    await product.update({ stock: newStock });
    res.json({ message: 'Stock updated successfully', product });
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
});

module.exports = router;