import React, { useState, useEffect } from 'react';
import { Card, Row, Col, Spinner, Badge } from 'react-bootstrap';
import axios from 'axios';
import './CashBoxWidget.css';

const CashBoxWidget = () => {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchTodayData();
  }, []);

  const fetchTodayData = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/cash-reconciliation/today', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setData(response.data);
    } catch (error) {
      console.error('Error fetching today\'s cash reconciliation:', error);
    } finally {
      setLoading(false);
    }
  };

  const getVarianceStatus = (variance) => {
    if (variance === 0) return { text: 'Balanced', variant: 'success' };
    if (variance < 0) return { text: `Short ৳${Math.abs(variance).toLocaleString()}`, variant: 'danger' };
    return { text: `Over ৳${variance.toLocaleString()}`, variant: 'warning' };
  };

  if (loading) {
    return (
      <Card className="cash-box-widget">
        <Card.Header>
          <h6 className="mb-0">
            <i className="fas fa-cash-register me-2"></i>
            Cash Box - Today
          </h6>
        </Card.Header>
        <Card.Body className="text-center">
          <Spinner animation="border" size="sm" />
        </Card.Body>
      </Card>
    );
  }

  if (!data) {
    return (
      <Card className="cash-box-widget">
        <Card.Header>
          <h6 className="mb-0">
            <i className="fas fa-cash-register me-2"></i>
            Cash Box - Today
          </h6>
        </Card.Header>
        <Card.Body className="text-center">
          <p className="text-muted mb-0">No reconciliation data for today</p>
          <small className="text-muted">Start your daily cash reconciliation</small>
        </Card.Body>
      </Card>
    );
  }

  const varianceStatus = getVarianceStatus(parseFloat(data.variance || 0));

  return (
    <Card className="cash-box-widget">
      <Card.Header>
        <h6 className="mb-0">
          <i className="fas fa-cash-register me-2"></i>
          Cash Box - Today
        </h6>
      </Card.Header>
      <Card.Body>
        <Row className="text-center">
          <Col>
            <div className="widget-stat">
              <div className="stat-value">৳{parseFloat(data.opening_total || 0).toLocaleString()}</div>
              <div className="stat-label">Opening Cash</div>
            </div>
          </Col>
          <Col>
            <div className="widget-stat">
              <div className="stat-value">৳{parseFloat(data.closing_total || 0).toLocaleString()}</div>
              <div className="stat-label">Closing Cash</div>
            </div>
          </Col>
        </Row>
        <Row className="mt-3">
          <Col className="text-center">
            <div className="variance-section">
              <div className="variance-label">Daily Variance</div>
              <div className={`variance-value ${varianceStatus.variant}`}>
                ৳{parseFloat(data.variance || 0).toLocaleString()}
              </div>
              <Badge bg={varianceStatus.variant} className="mt-1">
                {varianceStatus.text}
              </Badge>
            </div>
          </Col>
        </Row>
        <div className="widget-footer mt-3">
          <small className="text-muted">
            <i className="fas fa-clock me-1"></i>
            Last updated: {data.updatedAt ? new Date(data.updatedAt).toLocaleTimeString() : 'Not updated'}
          </small>
        </div>
      </Card.Body>
    </Card>
  );
};

export default CashBoxWidget;