import React, { useState, useEffect } from 'react';
import { Card, Badge, Row, Col, Spinner, Alert } from 'react-bootstrap';

const DSOAllocationBadge = () => {
  const [allocations, setAllocations] = useState({
    cardsAllocated: 0,
    simsAllocated: 0,
    kitsAllocated: 0,
    cardsSold: 0,
    simsSold: 0,
    kitsSold: 0
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    fetchTodayAllocations();
  }, []);

  const fetchTodayAllocations = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/dso/allocations/today');
      const data = await response.json();
      
      if (response.ok) {
        setAllocations(data);
        setError(null);
      } else {
        setError(data.message || 'Failed to fetch allocations');
      }
    } catch (error) {
      console.error('Error fetching today allocations:', error);
      setError('Network error occurred');
    } finally {
      setLoading(false);
    }
  };

  const getStockStatus = (allocated, sold) => {
    const remaining = allocated - sold;
    const percentage = allocated > 0 ? (remaining / allocated) * 100 : 0;
    
    if (percentage > 50) return 'success';
    if (percentage > 20) return 'warning';
    return 'danger';
  };

  if (loading) {
    return (
      <Card className="mb-3 bg-light">
        <Card.Body className="py-2 text-center">
          <Spinner animation="border" size="sm" className="me-2" />
          Loading DSO allocations...
        </Card.Body>
      </Card>
    );
  }

  if (error) {
    return (
      <Alert variant="warning" className="mb-3 py-2">
        <small>
          <i className="fas fa-exclamation-triangle me-2"></i>
          {error}
        </small>
      </Alert>
    );
  }

  const hasAllocations = allocations.cardsAllocated > 0 || 
                        allocations.simsAllocated > 0 || 
                        allocations.kitsAllocated > 0;

  if (!hasAllocations) {
    return (
      <Alert variant="info" className="mb-3 py-2">
        <small>
          <i className="fas fa-info-circle me-2"></i>
          No DSO allocations for today
        </small>
      </Alert>
    );
  }

  return (
    <Card className="mb-3 bg-light border-primary">
      <Card.Body className="py-2">
        <div className="d-flex justify-content-between align-items-center mb-2">
          <h6 className="mb-0 text-primary">
            <i className="fas fa-share-alt me-2"></i>
            Today's DSO Stock
          </h6>
          <small className="text-muted">
            {new Date().toLocaleDateString()}
          </small>
        </div>
        
        <Row className="g-2">
          {allocations.cardsAllocated > 0 && (
            <Col xs={4}>
              <div className="text-center">
                <Badge 
                  bg={getStockStatus(allocations.cardsAllocated, allocations.cardsSold)}
                  className="d-block mb-1"
                >
                  <i className="fas fa-credit-card me-1"></i>
                  Cards
                </Badge>
                <small className="text-muted d-block">
                  {allocations.cardsAllocated - allocations.cardsSold} / {allocations.cardsAllocated}
                </small>
              </div>
            </Col>
          )}
          
          {allocations.simsAllocated > 0 && (
            <Col xs={4}>
              <div className="text-center">
                <Badge 
                  bg={getStockStatus(allocations.simsAllocated, allocations.simsSold)}
                  className="d-block mb-1"
                >
                  <i className="fas fa-sim-card me-1"></i>
                  SIMs
                </Badge>
                <small className="text-muted d-block">
                  {allocations.simsAllocated - allocations.simsSold} / {allocations.simsAllocated}
                </small>
              </div>
            </Col>
          )}
          
          {allocations.kitsAllocated > 0 && (
            <Col xs={4}>
              <div className="text-center">
                <Badge 
                  bg={getStockStatus(allocations.kitsAllocated, allocations.kitsSold)}
                  className="d-block mb-1"
                >
                  <i className="fas fa-box me-1"></i>
                  Kits
                </Badge>
                <small className="text-muted d-block">
                  {allocations.kitsAllocated - allocations.kitsSold} / {allocations.kitsAllocated}
                </small>
              </div>
            </Col>
          )}
        </Row>
        
        <div className="mt-2 pt-2 border-top">
          <small className="text-muted">
            <i className="fas fa-info-circle me-1"></i>
            Available / Total allocated stock
          </small>
        </div>
      </Card.Body>
    </Card>
  );
};

export default DSOAllocationBadge;