import React, { useState, useEffect } from 'react';
import { Modal, Button, Alert, Card, Row, Col, Badge, Form } from 'react-bootstrap';

const DSOStockSelectionModal = ({ show, onHide, cartItems, onStockSelectionComplete }) => {
  const [stockSelections, setStockSelections] = useState({});
  const [todayAllocations, setTodayAllocations] = useState({
    cardsAllocated: 0,
    simsAllocated: 0,
    kitsAllocated: 0,
    cardTypesAllocated: []
  });
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (show) {
      fetchTodayAllocations();
      initializeStockSelections();
    }
  }, [show, cartItems]);

  const fetchTodayAllocations = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/dso/allocations/today');
      const data = await response.json();
      
      if (response.ok) {
        setTodayAllocations(data);
      }
    } catch (error) {
      console.error('Error fetching today allocations:', error);
    } finally {
      setLoading(false);
    }
  };

  const initializeStockSelections = () => {
    const selections = {};
    cartItems.forEach(item => {
      // Check if item is a DSO-related product (cards, SIMs, kits)
      const itemType = getDSOItemType(item);
      if (itemType) {
        selections[item.id] = {
          itemType,
          useAllocated: true, // Default to using allocated stock
          quantity: item.quantity
        };
      }
    });
    setStockSelections(selections);
  };

  const getDSOItemType = (item) => {
    const name = item.name.toLowerCase();
    if (name.includes('card') || name.includes('sim card')) {
      return 'cards';
    } else if (name.includes('sim') && !name.includes('card')) {
      return 'sims';
    } else if (name.includes('kit')) {
      return 'kits';
    }
    return null;
  };

  const handleStockSelectionChange = (itemId, useAllocated) => {
    setStockSelections(prev => ({
      ...prev,
      [itemId]: {
        ...prev[itemId],
        useAllocated
      }
    }));
  };

  const handleConfirm = () => {
    // Validate that allocated stock is sufficient
    const validation = validateStockSelections();
    if (!validation.valid) {
      alert(validation.message);
      return;
    }

    onStockSelectionComplete(stockSelections);
    onHide();
  };

  const validateStockSelections = () => {
    const allocatedUsage = {
      cards: 0,
      sims: 0,
      kits: 0
    };

    // Calculate total allocated stock usage
    Object.values(stockSelections).forEach(selection => {
      if (selection.useAllocated) {
        allocatedUsage[selection.itemType] += selection.quantity;
      }
    });

    // Check if we have enough allocated stock
    if (allocatedUsage.cards > todayAllocations.cardsAllocated) {
      return {
        valid: false,
        message: `Not enough allocated cards. Requested: ${allocatedUsage.cards}, Available: ${todayAllocations.cardsAllocated}`
      };
    }

    if (allocatedUsage.sims > todayAllocations.simsAllocated) {
      return {
        valid: false,
        message: `Not enough allocated SIMs. Requested: ${allocatedUsage.sims}, Available: ${todayAllocations.simsAllocated}`
      };
    }

    if (allocatedUsage.kits > todayAllocations.kitsAllocated) {
      return {
        valid: false,
        message: `Not enough allocated kits. Requested: ${allocatedUsage.kits}, Available: ${todayAllocations.kitsAllocated}`
      };
    }

    return { valid: true };
  };

  const getDSOItems = () => {
    return cartItems.filter(item => getDSOItemType(item) !== null);
  };

  const dsoItems = getDSOItems();

  if (dsoItems.length === 0) {
    // No DSO items in cart, skip this modal
    if (show) {
      onStockSelectionComplete({});
      onHide();
    }
    return null;
  }

  return (
    <Modal show={show} onHide={onHide} size="lg">
      <Modal.Header closeButton>
        <Modal.Title>
          <i className="fas fa-share-alt me-2"></i>
          Select Stock Source
        </Modal.Title>
      </Modal.Header>
      <Modal.Body>
        <Alert variant="info">
          <i className="fas fa-info-circle me-2"></i>
          Your cart contains DSO-related items. Please choose whether to use allocated stock or general stock for each item.
        </Alert>

        {/* Today's Allocations Summary */}
        <Card className="mb-3 bg-light">
          <Card.Body className="py-2">
            <h6 className="mb-2">Today's Available Allocations:</h6>
            <div className="d-flex gap-3">
              <Badge bg="primary">
                <i className="fas fa-credit-card me-1"></i>
                {todayAllocations.cardsAllocated} Cards
              </Badge>
              <Badge bg="success">
                <i className="fas fa-sim-card me-1"></i>
                {todayAllocations.simsAllocated} SIMs
              </Badge>
              <Badge bg="warning">
                <i className="fas fa-box me-1"></i>
                {todayAllocations.kitsAllocated} Kits
              </Badge>
            </div>
          </Card.Body>
        </Card>

        {/* Stock Selection for Each DSO Item */}
        <div className="stock-selection-items">
          {dsoItems.map(item => {
            const itemType = getDSOItemType(item);
            const selection = stockSelections[item.id];
            
            return (
              <Card key={item.id} className="mb-3">
                <Card.Body>
                  <Row className="align-items-center">
                    <Col md={4}>
                      <h6 className="mb-1">{item.name}</h6>
                      <small className="text-muted">
                        Quantity: {item.quantity} | Type: {itemType}
                      </small>
                    </Col>
                    <Col md={8}>
                      <div className="d-flex gap-2">
                        <Form.Check
                          type="radio"
                          id={`allocated-${item.id}`}
                          name={`stock-${item.id}`}
                          label={
                            <span>
                              <Badge bg="info" className="me-2">Allocated</Badge>
                              Use DSO Allocated Stock
                            </span>
                          }
                          checked={selection?.useAllocated === true}
                          onChange={() => handleStockSelectionChange(item.id, true)}
                        />
                        <Form.Check
                          type="radio"
                          id={`general-${item.id}`}
                          name={`stock-${item.id}`}
                          label={
                            <span>
                              <Badge bg="secondary" className="me-2">General</Badge>
                              Use General Stock
                            </span>
                          }
                          checked={selection?.useAllocated === false}
                          onChange={() => handleStockSelectionChange(item.id, false)}
                        />
                      </div>
                    </Col>
                  </Row>
                </Card.Body>
              </Card>
            );
          })}
        </div>

        {/* Usage Summary */}
        <Card className="bg-light">
          <Card.Body className="py-2">
            <h6 className="mb-2">Stock Usage Summary:</h6>
            <Row>
              <Col md={4}>
                <small>
                  <strong>Cards from Allocated:</strong> {
                    Object.values(stockSelections)
                      .filter(s => s.itemType === 'cards' && s.useAllocated)
                      .reduce((sum, s) => sum + s.quantity, 0)
                  } / {todayAllocations.cardsAllocated}
                </small>
              </Col>
              <Col md={4}>
                <small>
                  <strong>SIMs from Allocated:</strong> {
                    Object.values(stockSelections)
                      .filter(s => s.itemType === 'sims' && s.useAllocated)
                      .reduce((sum, s) => sum + s.quantity, 0)
                  } / {todayAllocations.simsAllocated}
                </small>
              </Col>
              <Col md={4}>
                <small>
                  <strong>Kits from Allocated:</strong> {
                    Object.values(stockSelections)
                      .filter(s => s.itemType === 'kits' && s.useAllocated)
                      .reduce((sum, s) => sum + s.quantity, 0)
                  } / {todayAllocations.kitsAllocated}
                </small>
              </Col>
            </Row>
          </Card.Body>
        </Card>
      </Modal.Body>
      <Modal.Footer>
        <Button variant="secondary" onClick={onHide}>
          Cancel
        </Button>
        <Button variant="primary" onClick={handleConfirm} disabled={loading}>
          <i className="fas fa-check me-2"></i>
          Confirm Stock Selection
        </Button>
      </Modal.Footer>
    </Modal>
  );
};

export default DSOStockSelectionModal;