import React, { useState, useEffect, useRef } from 'react';
import { Card, Form, Button, ListGroup, Badge, Modal, Row, Col, Alert } from 'react-bootstrap';
import { FaPlus, FaEdit, FaTrash, FaCheck, FaTimes, FaStickyNote, FaTasks, FaMicrophone, FaMicrophoneSlash } from 'react-icons/fa';
import axios from 'axios';

const Notepad = () => {
  const [notes, setNotes] = useState([]);
  const [todos, setTodos] = useState([]);
  const [activeTab, setActiveTab] = useState('notes');
  const [showModal, setShowModal] = useState(false);
  const [editingItem, setEditingItem] = useState(null);
  const [formData, setFormData] = useState({
    title: '',
    content: '',
    priority: 'medium',
    type: 'note'
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [isListening, setIsListening] = useState(false);
  const [voiceField, setVoiceField] = useState(''); // 'title' or 'content'
  const [speechSupported, setSpeechSupported] = useState(false);

  const recognitionRef = useRef(null);
  const API_BASE_URL = 'http://localhost:5000/api';

  useEffect(() => {
    fetchTodayData();
    initializeSpeechRecognition();
  }, []);

  const initializeSpeechRecognition = () => {
    if ('webkitSpeechRecognition' in window || 'SpeechRecognition' in window) {
      const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
      recognitionRef.current = new SpeechRecognition();
      
      recognitionRef.current.continuous = false;
      recognitionRef.current.interimResults = false;
      recognitionRef.current.lang = 'en-US';
      
      recognitionRef.current.onstart = () => {
        setIsListening(true);
      };
      
      recognitionRef.current.onresult = (event) => {
        const transcript = event.results[0][0].transcript;
        if (voiceField === 'title') {
          setFormData(prev => ({ ...prev, title: prev.title + transcript }));
        } else if (voiceField === 'content') {
          setFormData(prev => ({ ...prev, content: prev.content + transcript }));
        }
        setIsListening(false);
        setVoiceField(null);
      };
      
      recognitionRef.current.onerror = (event) => {
        console.error('Speech recognition error:', event.error);
        setIsListening(false);
        setVoiceField(null);
        
        // Provide user-friendly error messages
        let errorMessage = 'Voice input failed. ';
        switch (event.error) {
          case 'not-allowed':
            errorMessage += 'Please allow microphone access in your browser settings and try again.';
            break;
          case 'no-speech':
            errorMessage += 'No speech was detected. Please try speaking again.';
            break;
          case 'audio-capture':
            errorMessage += 'Microphone not found or not working.';
            break;
          case 'network':
            errorMessage += 'Network error occurred. Please check your connection.';
            break;
          default:
            errorMessage += 'Please try again or type manually.';
        }
        setError(errorMessage);
        
        // Clear error after 5 seconds
        setTimeout(() => setError(''), 5000);
      };
      
      recognitionRef.current.onend = () => {
        setIsListening(false);
        setVoiceField(null);
      };
      
      setSpeechSupported(true);
    } else {
      setSpeechSupported(false);
      console.warn('Speech recognition not supported in this browser');
    }
  };

  const requestMicrophonePermission = async () => {
    try {
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
      stream.getTracks().forEach(track => track.stop()); // Stop the stream immediately
      return true;
    } catch (error) {
      console.error('Microphone permission denied:', error);
      setError('Microphone access is required for voice input. Please allow microphone access in your browser settings.');
      setTimeout(() => setError(''), 5000);
      return false;
    }
  };

  const startVoiceInput = async (field) => {
    if (!speechSupported || !recognitionRef.current) {
      setError('Voice input is not supported in this browser.');
      setTimeout(() => setError(''), 3000);
      return;
    }
    
    if (isListening) {
      recognitionRef.current.stop();
      return;
    }
    
    // Request microphone permission first
    const hasPermission = await requestMicrophonePermission();
    if (!hasPermission) return;
    
    try {
      setVoiceField(field);
      setError(''); // Clear any previous errors
      recognitionRef.current.start();
    } catch (error) {
      console.error('Failed to start voice recognition:', error);
      setIsListening(false);
      setVoiceField(null);
      setError('Failed to start voice input. Please try again.');
      setTimeout(() => setError(''), 3000);
    }
  };

  const stopVoiceInput = () => {
    if (recognitionRef.current && isListening) {
      recognitionRef.current.stop();
    }
  };

  const getAuthHeaders = () => {
    const token = localStorage.getItem('token');
    return {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    };
  };

  const fetchTodayData = async () => {
    setLoading(true);
    try {
      const response = await axios.get(`${API_BASE_URL}/notes/today`, getAuthHeaders());
      const data = response.data;
      
      setNotes(data.filter(item => item.type === 'note'));
      setTodos(data.filter(item => item.type === 'todo'));
    } catch (error) {
      console.error('Error fetching data:', error);
      setError('Failed to load notes and todos');
    } finally {
      setLoading(false);
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    stopVoiceInput(); // Stop any ongoing voice input
    setLoading(true);
    
    try {
      const dataToSend = {
        ...formData,
        date: new Date().toISOString().split('T')[0]
      };

      if (editingItem) {
        await axios.put(`${API_BASE_URL}/notes/${editingItem.id}`, dataToSend, getAuthHeaders());
      } else {
        await axios.post(`${API_BASE_URL}/notes`, dataToSend, getAuthHeaders());
      }
      
      closeModal();
      fetchTodayData();
    } catch (error) {
      console.error('Error saving item:', error);
      setError('Failed to save item');
    } finally {
      setLoading(false);
    }
  };

  const handleDelete = async (id) => {
    if (window.confirm('Are you sure you want to delete this item?')) {
      try {
        await axios.delete(`${API_BASE_URL}/notes/${id}`, getAuthHeaders());
        fetchTodayData();
      } catch (error) {
        console.error('Error deleting item:', error);
        setError('Failed to delete item');
      }
    }
  };

  const handleToggleTodo = async (id) => {
    try {
      await axios.patch(`${API_BASE_URL}/notes/${id}/toggle`, {}, getAuthHeaders());
      fetchTodayData();
    } catch (error) {
      console.error('Error toggling todo:', error);
      setError('Failed to update todo status');
    }
  };

  const openModal = (type, item = null) => {
    if (item) {
      setEditingItem(item);
      setFormData({
        title: item.title,
        content: item.content,
        priority: item.priority,
        type: item.type
      });
    } else {
      setEditingItem(null);
      setFormData({
        title: '',
        content: '',
        priority: 'medium',
        type: type
      });
    }
    setShowModal(true);
  };

  const closeModal = () => {
    stopVoiceInput();
    setShowModal(false);
    setEditingItem(null);
    setFormData({ title: '', content: '', priority: 'medium', type: 'note' });
  };

  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'high': return 'danger';
      case 'medium': return 'warning';
      case 'low': return 'success';
      default: return 'secondary';
    }
  };

  return (
    <Card className="h-100">
      <Card.Header className="d-flex justify-content-between align-items-center">
        <div>
          <Button
            variant={activeTab === 'notes' ? 'primary' : 'outline-primary'}
            size="sm"
            className="me-2"
            onClick={() => setActiveTab('notes')}
          >
            <FaStickyNote className="me-1" />
            Notes ({notes.length})
          </Button>
          <Button
            variant={activeTab === 'todos' ? 'primary' : 'outline-primary'}
            size="sm"
            onClick={() => setActiveTab('todos')}
          >
            <FaTasks className="me-1" />
            Todos ({todos.length})
          </Button>
        </div>
        <Button
          variant="success"
          size="sm"
          onClick={() => openModal(activeTab === 'notes' ? 'note' : 'todo')}
        >
          <FaPlus />
        </Button>
      </Card.Header>

      <Card.Body style={{ maxHeight: '400px', overflowY: 'auto' }}>
        {error && (
          <Alert variant="danger" dismissible onClose={() => setError('')}>
            {error}
          </Alert>
        )}

        {loading ? (
          <div className="text-center">Loading...</div>
        ) : (
          <ListGroup variant="flush">
            {activeTab === 'notes' && notes.map(note => (
              <ListGroup.Item key={note.id} className="px-0">
                <div className="d-flex justify-content-between align-items-start">
                  <div className="flex-grow-1">
                    <h6 className="mb-1">{note.title}</h6>
                    <p className="mb-1 text-muted small">{note.content}</p>
                    <Badge bg={getPriorityColor(note.priority)} className="me-2">
                      {note.priority}
                    </Badge>
                    <small className="text-muted">
                      {new Date(note.createdAt).toLocaleTimeString()}
                    </small>
                  </div>
                  <div>
                    <Button
                      variant="outline-primary"
                      size="sm"
                      className="me-1"
                      onClick={() => openModal('note', note)}
                    >
                      <FaEdit />
                    </Button>
                    <Button
                      variant="outline-danger"
                      size="sm"
                      onClick={() => handleDelete(note.id)}
                    >
                      <FaTrash />
                    </Button>
                  </div>
                </div>
              </ListGroup.Item>
            ))}

            {activeTab === 'todos' && todos.map(todo => (
              <ListGroup.Item key={todo.id} className="px-0">
                <div className="d-flex justify-content-between align-items-start">
                  <div className="flex-grow-1">
                    <div className="d-flex align-items-center">
                      <Button
                        variant={todo.isCompleted ? 'success' : 'outline-secondary'}
                        size="sm"
                        className="me-2"
                        onClick={() => handleToggleTodo(todo.id)}
                      >
                        {todo.isCompleted ? <FaCheck /> : <FaTimes />}
                      </Button>
                      <div className={todo.isCompleted ? 'text-decoration-line-through text-muted' : ''}>
                        <h6 className="mb-1">{todo.title}</h6>
                        <p className="mb-1 small">{todo.content}</p>
                      </div>
                    </div>
                    <div className="mt-1">
                      <Badge bg={getPriorityColor(todo.priority)} className="me-2">
                        {todo.priority}
                      </Badge>
                      <small className="text-muted">
                        {new Date(todo.createdAt).toLocaleTimeString()}
                      </small>
                    </div>
                  </div>
                  <div>
                    <Button
                      variant="outline-primary"
                      size="sm"
                      className="me-1"
                      onClick={() => openModal('todo', todo)}
                    >
                      <FaEdit />
                    </Button>
                    <Button
                      variant="outline-danger"
                      size="sm"
                      onClick={() => handleDelete(todo.id)}
                    >
                      <FaTrash />
                    </Button>
                  </div>
                </div>
              </ListGroup.Item>
            ))}

            {((activeTab === 'notes' && notes.length === 0) || 
              (activeTab === 'todos' && todos.length === 0)) && (
              <ListGroup.Item className="text-center text-muted">
                No {activeTab} for today. Click + to add one!
              </ListGroup.Item>
            )}
          </ListGroup>
        )}
      </Card.Body>

      {/* Add/Edit Modal */}
      <Modal show={showModal} onHide={closeModal}>
        <Modal.Header closeButton>
          <Modal.Title>
            {editingItem ? 'Edit' : 'Add'} {formData.type === 'note' ? 'Note' : 'Todo'}
            {speechSupported && (
              <small className="text-muted ms-2">
                (🎤 Voice input available - Click microphone buttons to speak)
              </small>
            )}
          </Modal.Title>
        </Modal.Header>
        <Form onSubmit={handleSubmit}>
          <Modal.Body>
            <Form.Group className="mb-3">
              <Form.Label>Title</Form.Label>
              <div className="d-flex">
                <Form.Control
                  type="text"
                  value={formData.title}
                  onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                  placeholder={`Enter ${formData.type} title`}
                  required
                  className="me-2"
                />
                {speechSupported && (
                  <Button
                    variant={isListening && voiceField === 'title' ? 'danger' : 'outline-primary'}
                    size="sm"
                    onClick={() => startVoiceInput('title')}
                    disabled={loading}
                    title={isListening && voiceField === 'title' ? 'Stop recording' : 'Voice input for title'}
                  >
                    {isListening && voiceField === 'title' ? <FaMicrophoneSlash /> : <FaMicrophone />}
                  </Button>
                )}
              </div>
              {isListening && voiceField === 'title' && (
                <small className="text-muted">🎤 Listening for title...</small>
              )}
            </Form.Group>

            <Form.Group className="mb-3">
              <Form.Label>Content</Form.Label>
              <div className="d-flex">
                <Form.Control
                  as="textarea"
                  rows={3}
                  value={formData.content}
                  onChange={(e) => setFormData({ ...formData, content: e.target.value })}
                  placeholder={`Enter ${formData.type} content`}
                  required
                  className="me-2"
                />
                {speechSupported && (
                  <Button
                    variant={isListening && voiceField === 'content' ? 'danger' : 'outline-primary'}
                    size="sm"
                    onClick={() => startVoiceInput('content')}
                    disabled={loading}
                    title={isListening && voiceField === 'content' ? 'Stop recording' : 'Voice input for content'}
                    style={{ alignSelf: 'flex-start' }}
                  >
                    {isListening && voiceField === 'content' ? <FaMicrophoneSlash /> : <FaMicrophone />}
                  </Button>
                )}
              </div>
              {isListening && voiceField === 'content' && (
                <small className="text-muted">🎤 Listening for content...</small>
              )}
            </Form.Group>

            <Form.Group className="mb-3">
              <Form.Label>Priority</Form.Label>
              <Form.Select
                value={formData.priority}
                onChange={(e) => setFormData({ ...formData, priority: e.target.value })}
              >
                <option value="low">Low</option>
                <option value="medium">Medium</option>
                <option value="high">High</option>
              </Form.Select>
            </Form.Group>

            {speechSupported && (
              <div className="alert alert-info py-2 px-3 mb-3">
                <small>
                  <strong>🎤 Voice Input Tips:</strong><br/>
                  • Click the microphone button and speak clearly<br/>
                  • If voice input doesn't work, allow microphone access in browser settings<br/>
                  • For Chrome: Click the 🔒 icon in address bar → Allow microphone<br/>
                  • Voice input works best in quiet environments
                </small>
              </div>
            )}
          </Modal.Body>
          <Modal.Footer>
            <Button variant="secondary" onClick={closeModal}>
              Cancel
            </Button>
            <Button variant="primary" type="submit" disabled={loading || isListening}>
              {loading ? 'Saving...' : 'Save'}
            </Button>
          </Modal.Footer>
        </Form>
      </Modal>
    </Card>
  );
};

export default Notepad;