import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Spinner, Alert } from 'react-bootstrap';
import { FaShoppingCart, FaBoxes, FaUsers, FaChartLine, FaMoneyBillWave, FaExclamationTriangle, FaMobileAlt, FaUserTie, FaStore, FaCreditCard } from 'react-icons/fa';
import axios from 'axios';
import CashBoxWidget from '../components/CashBoxWidget';
import Notepad from '../components/Notepad';
import BkashSmsWidget from '../components/BkashSmsWidget';

const Dashboard = () => {
  const [dashboardData, setDashboardData] = useState({
    todaySales: { count: 0, revenue: 0 },
    recentSales: [],
    totalProducts: 0,
    totalCustomers: 0,
    totalDueAmount: 0,
    lowStockProducts: 0,
    bkashAgent: { opening: 0, closing: 0, profit: 0, srReceived: 0, srGiven: 0 },
    bkashPersonal: { opening: 0, closing: 0, transactions: 0 },
    bkashMerchant: { totalAccounts: 0, totalBalance: 0, todayTransactions: 0, totalTransactions: 0 },
    accountNumbers: { totalAccounts: 0, activeAccounts: 0 },
    mobileRecharge: { 
      totalOpeningBalance: 0, 
      totalClosingBalance: 0, 
      calculatedEndBalance: 0,
      totalRecharge: 0, 
      totalCommission: 0, 
      totalTransactions: 0,
      operatorsOpened: 0,
      operatorsClosed: 0,
      totalOperators: 3,
      status: { allOpened: false, allClosed: false, partiallyOpened: false }
    }
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    fetchDashboardData();
  }, []);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const headers = { Authorization: `Bearer ${token}` };

      // Fetch today's sales summary
      const salesResponse = await axios.get('http://localhost:5000/api/sales/summary?period=today', { headers });
      
      // Fetch recent sales (last 5 sales)
      const recentSalesResponse = await axios.get('http://localhost:5000/api/sales?limit=5', { headers });
      
      // Fetch total products count
      const productsResponse = await axios.get('http://localhost:5000/api/products', { headers });
      
      // Fetch total customers count
      const customersResponse = await axios.get('http://localhost:5000/api/customers', { headers });
      
      // Fetch customers with due balance
      const dueCustomersResponse = await axios.get('http://localhost:5000/api/customers/due', { headers });
      
      // Calculate total due amount
      const totalDue = dueCustomersResponse.data.reduce((sum, customer) => sum + parseFloat(customer.dueBalance || 0), 0);
      
      // Count low stock products (assuming stock < 10 is low)
      const lowStockCount = productsResponse.data.filter(product => product.stock < 10).length;

      // Fetch Bkash data
      let bkashAgentData = { opening: 0, closing: 0, profit: 0, srReceived: 0, srGiven: 0 };
      let bkashPersonalData = { opening: 0, closing: 0, transactions: 0 };
      let bkashMerchantData = { totalAccounts: 0, totalBalance: 0, todayTransactions: 0, totalTransactions: 0 };
      let accountNumbersData = { totalAccounts: 0, activeAccounts: 0 };
      
      try {
        const bkashAgentResponse = await axios.get('http://localhost:5000/api/bkash-agent/today', { headers });
        if (bkashAgentResponse.data) {
          const agent = bkashAgentResponse.data;
          bkashAgentData = {
            opening: agent.openingBalance || 0,
            closing: agent.closingBalance || 0,
            profit: (agent.closingBalance || 0) - (agent.openingBalance || 0),
            srReceived: agent.srReceived || 0,
            srGiven: agent.srGiven || 0
          };
        }
      } catch (error) {
        console.log('Bkash agent data not available');
      }

      try {
        const bkashPersonalResponse = await axios.get('http://localhost:5000/api/bkash-personal/today', { headers });
        if (bkashPersonalResponse.data) {
          const personal = bkashPersonalResponse.data;
          bkashPersonalData = {
            opening: personal.openingBalance || 0,
            closing: personal.closingBalance || 0,
            transactions: 0 // Will be fetched separately if needed
          };
        }

        // Fetch today's transactions count
        const today = new Date().toISOString().split('T')[0];
        const transactionsResponse = await axios.get(`http://localhost:5000/api/bkash-personal/${today}/transactions`, { headers });
        bkashPersonalData.transactions = transactionsResponse.data.length || 0;
      } catch (error) {
        console.log('Bkash personal data not available');
      }

      // Fetch Bkash Merchant data
      try {
        const merchantAccountsResponse = await axios.get('http://localhost:5000/api/bkash-merchant', { headers });
        const merchantAccounts = merchantAccountsResponse.data || [];
        
        const totalBalance = merchantAccounts.reduce((sum, account) => sum + (account.balance || 0), 0);
        
        // Fetch today's merchant transactions
        let todayTransactions = 0;
        let totalTransactions = 0;
        
        for (const account of merchantAccounts) {
          try {
            const transResponse = await axios.get(`http://localhost:5000/api/bkash-merchant/${account.id}/transactions`, { headers });
            const transactions = transResponse.data || [];
            totalTransactions += transactions.length;
            
            const todayTrans = transactions.filter(t => 
              new Date(t.createdAt).toDateString() === new Date().toDateString()
            );
            todayTransactions += todayTrans.length;
          } catch (error) {
            console.log(`Error fetching transactions for merchant account ${account.id}`);
          }
        }
        
        bkashMerchantData = {
          totalAccounts: merchantAccounts.length,
          totalBalance: totalBalance,
          todayTransactions: todayTransactions,
          totalTransactions: totalTransactions
        };
      } catch (error) {
        console.log('Bkash merchant data not available');
      }

      // Fetch Account Numbers data
      try {
        const accountNumbersResponse = await axios.get('http://localhost:5000/api/bkash-accounts', { headers });
        const accounts = accountNumbersResponse.data || [];
        const activeAccounts = accounts.filter(account => account.status === 'active');
        
        accountNumbersData = {
          totalAccounts: accounts.length,
          activeAccounts: activeAccounts.length
        };
      } catch (error) {
        console.log('Account numbers data not available');
      }

      // Fetch Mobile Recharge data
      let mobileRechargeData = { 
        totalOpeningBalance: 0, 
        totalClosingBalance: 0, 
        calculatedEndBalance: 0,
        totalRecharge: 0, 
        totalCommission: 0, 
        totalTransactions: 0,
        operatorsOpened: 0,
        operatorsClosed: 0,
        totalOperators: 3,
        status: { allOpened: false, allClosed: false, partiallyOpened: false }
      };
      
      try {
        const mobileRechargeResponse = await axios.get('http://localhost:5000/api/mobile-recharge/dashboard-summary', { headers });
        if (mobileRechargeResponse.data) {
          mobileRechargeData = mobileRechargeResponse.data;
        }
      } catch (error) {
        console.log('Mobile recharge data not available');
      }

      setDashboardData({
        todaySales: {
          count: salesResponse.data.totalSales || 0,
          revenue: salesResponse.data.revenue || 0
        },
        recentSales: recentSalesResponse.data || [],
        totalProducts: productsResponse.data.length || 0,
        totalCustomers: customersResponse.data.length || 0,
        totalDueAmount: totalDue,
        lowStockProducts: lowStockCount,
        bkashAgent: bkashAgentData,
        bkashPersonal: bkashPersonalData,
        bkashMerchant: bkashMerchantData,
        accountNumbers: accountNumbersData,
        mobileRecharge: mobileRechargeData
      });
      
      setError('');
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
      console.error('Error response:', error.response?.data);
      console.error('Error status:', error.response?.status);
      setError(`Failed to load dashboard data: ${error.response?.data?.message || error.message}`);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount);
  };

  if (loading) {
    return (
      <Container className="dashboard-container">
        <div className="text-center py-5">
          <Spinner animation="border" variant="primary" />
          <p className="mt-2">Loading dashboard...</p>
        </div>
      </Container>
    );
  }

  return (
    <Container className="dashboard-container">
      <h1 className="mb-4 text-center">Dashboard</h1>
      
      {error && (
        <Alert variant="danger" dismissible onClose={() => setError('')}>
          {error}
        </Alert>
      )}
      
      <Row className="mb-4 justify-content-center">
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card">
            <Card.Body>
              <FaShoppingCart size={40} className="text-primary mb-2" />
              <Card.Title>Sales Today</Card.Title>
              <h3 className="text-success">{formatCurrency(dashboardData.todaySales.revenue)}</h3>
              <small className="text-muted">{dashboardData.todaySales.count} transactions</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card">
            <Card.Body>
              <FaBoxes size={40} className="text-info mb-2" />
              <Card.Title>Products</Card.Title>
              <h3 className="text-info">{dashboardData.totalProducts}</h3>
              {dashboardData.lowStockProducts > 0 && (
                <small className="text-warning">
                  <FaExclamationTriangle className="me-1" />
                  {dashboardData.lowStockProducts} low stock
                </small>
              )}
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card">
            <Card.Body>
              <FaUsers size={40} className="text-warning mb-2" />
              <Card.Title>Customers</Card.Title>
              <h3 className="text-warning">{dashboardData.totalCustomers}</h3>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card">
            <Card.Body>
              <FaMoneyBillWave size={40} className="text-danger mb-2" />
              <Card.Title>Due Amount</Card.Title>
              <h3 className="text-danger">{formatCurrency(dashboardData.totalDueAmount)}</h3>
              <small className="text-muted">Outstanding payments</small>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Bkash Dashboard Cards */}
      <Row className="mb-4 justify-content-center">
        <Col xs={12} className="mb-3">
          <h4 className="text-center">
            <FaMobileAlt className="me-2 text-primary" />
            Bkash Agent Account Summary
          </h4>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-primary">
            <Card.Body>
              <FaUserTie size={40} className="text-primary mb-2" />
              <Card.Title>Agent Opening</Card.Title>
              <h3 className="text-primary">৳{dashboardData.bkashAgent.opening.toLocaleString()}</h3>
              <small className="text-muted">Today's opening balance</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-success">
            <Card.Body>
              <FaUserTie size={40} className="text-success mb-2" />
              <Card.Title>Agent Closing</Card.Title>
              <h3 className="text-success">৳{dashboardData.bkashAgent.closing.toLocaleString()}</h3>
              <small className="text-muted">Current closing balance</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-warning">
            <Card.Body>
              <FaMobileAlt size={40} className="text-warning mb-2" />
              <Card.Title>Daily Profit</Card.Title>
              <h3 className={`${dashboardData.bkashAgent.profit >= 0 ? 'text-success' : 'text-danger'}`}>
                ৳{dashboardData.bkashAgent.profit.toLocaleString()}
              </h3>
              <small className="text-muted">Agent profit today</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-info">
            <Card.Body>
              <FaMobileAlt size={40} className="text-info mb-2" />
              <Card.Title>SR Received</Card.Title>
              <h3 className="text-info">৳{dashboardData.bkashAgent.srReceived.toLocaleString()}</h3>
              <small className="text-muted">SR received today</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-danger">
            <Card.Body>
              <FaMobileAlt size={40} className="text-danger mb-2" />
              <Card.Title>SR Given</Card.Title>
              <h3 className="text-danger">৳{dashboardData.bkashAgent.srGiven.toLocaleString()}</h3>
              <small className="text-muted">SR given today</small>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Merchant and Account Numbers Dashboard Cards */}
      <Row className="mb-4 justify-content-center">
        <Col xs={12} className="mb-3">
          <h4 className="text-center">
            <FaStore className="me-2 text-success" />
            Merchant & Account Management
          </h4>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-success">
            <Card.Body>
              <FaStore size={40} className="text-success mb-2" />
              <Card.Title>Merchant Accounts</Card.Title>
              <h3 className="text-success">{dashboardData.bkashMerchant.totalAccounts}</h3>
              <small className="text-muted">Total merchant accounts</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-primary">
            <Card.Body>
              <FaMoneyBillWave size={40} className="text-primary mb-2" />
              <Card.Title>Merchant Balance</Card.Title>
              <h3 className="text-primary">৳{dashboardData.bkashMerchant.totalBalance.toLocaleString()}</h3>
              <small className="text-muted">Total merchant balance</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-warning">
            <Card.Body>
              <FaCreditCard size={40} className="text-warning mb-2" />
              <Card.Title>Today's Transactions</Card.Title>
              <h3 className="text-warning">{dashboardData.bkashMerchant.todayTransactions}</h3>
              <small className="text-muted">Merchant transactions today</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={3} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-info">
            <Card.Body>
              <FaCreditCard size={40} className="text-info mb-2" />
              <Card.Title>Account Numbers</Card.Title>
              <h3 className="text-info">{dashboardData.accountNumbers.activeAccounts}/{dashboardData.accountNumbers.totalAccounts}</h3>
              <small className="text-muted">Active/Total accounts</small>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Mobile Recharge Dashboard Cards */}
      <Row className="mb-4 justify-content-center">
        <Col xs={12} className="mb-3">
          <h4 className="text-center">
            <FaMobileAlt className="me-2 text-success" />
            Mobile Recharge Summary
          </h4>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-primary">
            <Card.Body>
              <FaMoneyBillWave size={40} className="text-primary mb-2" />
              <Card.Title>Opening Balance</Card.Title>
              <h3 className="text-primary">৳{dashboardData.mobileRecharge.totalOpeningBalance.toLocaleString()}</h3>
              <small className="text-muted">Total opening balance</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-success">
            <Card.Body>
              <FaMoneyBillWave size={40} className="text-success mb-2" />
              <Card.Title>Closing Balance</Card.Title>
              <h3 className="text-success">৳{dashboardData.mobileRecharge.totalClosingBalance.toLocaleString()}</h3>
              <small className="text-muted">Current closing balance</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-info">
            <Card.Body>
              <FaChartLine size={40} className="text-info mb-2" />
              <Card.Title>End-of-Day Balance</Card.Title>
              <h3 className="text-info">৳{dashboardData.mobileRecharge.calculatedEndBalance.toLocaleString()}</h3>
              <small className="text-muted">Calculated end balance</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-warning">
            <Card.Body>
              <FaMobileAlt size={40} className="text-warning mb-2" />
              <Card.Title>Total Recharge</Card.Title>
              <h3 className="text-warning">৳{dashboardData.mobileRecharge.totalRecharge.toLocaleString()}</h3>
              <small className="text-muted">Today's recharge amount</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-danger">
            <Card.Body>
              <FaMoneyBillWave size={40} className="text-danger mb-2" />
              <Card.Title>Commission</Card.Title>
              <h3 className="text-danger">৳{dashboardData.mobileRecharge.totalCommission.toLocaleString()}</h3>
              <small className="text-muted">Total commission earned</small>
            </Card.Body>
          </Card>
        </Col>
        <Col xs={12} sm={6} md={2} className="mb-3 mb-md-0">
          <Card className="text-center h-100 dashboard-card border-dark">
            <Card.Body>
              <FaShoppingCart size={40} className="text-dark mb-2" />
              <Card.Title>Transactions</Card.Title>
              <h3 className="text-dark">{dashboardData.mobileRecharge.totalTransactions}</h3>
              <small className="text-muted">
                {dashboardData.mobileRecharge.operatorsOpened}/{dashboardData.mobileRecharge.totalOperators} operators opened
              </small>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      <Row className="justify-content-center">
        {/* Right sidebar content - shown first on mobile */}
        <Col xs={12} md={6} lg={4} className="order-1 order-lg-2 mb-3">
          <div className="mb-3">
            <CashBoxWidget />
          </div>
          <div className="mb-3">
            <BkashSmsWidget />
          </div>
          <Card className="dashboard-card">
            <Card.Header>
              <h5>Quick Actions</h5>
            </Card.Header>
            <Card.Body className="quick-actions">
              <div className="d-grid gap-2">
                <button className="btn btn-primary">New Sale</button>
                <button className="btn btn-success">Add Product</button>
                <button className="btn btn-info">View Reports</button>
              </div>
            </Card.Body>
          </Card>

          {/* Daily Notepad */}
          <div className="mt-3">
            <Notepad />
          </div>
        </Col>
        
        {/* Main content - shown second on mobile */}
        <Col xs={12} md={12} lg={8} className="order-2 order-lg-1 mb-3 mb-lg-0">
          <Card className="h-100 dashboard-card">
            <Card.Header>
              <h5>Recent Sales</h5>
            </Card.Header>
            <Card.Body>
              {dashboardData.recentSales.length > 0 ? (
                <div className="table-responsive">
                  <table className="table table-sm">
                    <thead>
                      <tr>
                        <th>Date</th>
                        <th>Customer</th>
                        <th>Amount</th>
                        <th>Payment</th>
                      </tr>
                    </thead>
                    <tbody>
                      {dashboardData.recentSales.map((sale) => (
                        <tr key={sale.id}>
                          <td>{new Date(sale.createdAt).toLocaleDateString()}</td>
                          <td>{sale.Customer ? sale.Customer.name : 'Walk-in Customer'}</td>
                          <td>{formatCurrency(sale.totalAmount)}</td>
                          <td>
                            <span className={`badge ${sale.paymentMethod === 'cash' ? 'bg-success' : 'bg-info'}`}>
                              {sale.paymentMethod}
                            </span>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              ) : (
                <p className="text-muted">No recent sales found.</p>
              )}
            </Card.Body>
          </Card>
        </Col>
      </Row>
    </Container>
  );
};

export default Dashboard;