import React, { useState } from 'react';
import { Card, Form, Button, Alert, Container, Row, Col } from 'react-bootstrap';
import { useNavigate } from 'react-router-dom';
import { FaLock, FaUser, FaSignInAlt } from 'react-icons/fa';
import { useAuth } from '../../contexts/AuthContext';
import './Login.css';

const Login = () => {
  const navigate = useNavigate();
  const { login } = useAuth();
  const [credentials, setCredentials] = useState({
    email: '',
    password: ''
  });
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setCredentials({
      ...credentials,
      [name]: value
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setError('');
    setLoading(true);

    try {
      const result = await login(credentials.email, credentials.password);
      
      if (result.success) {
        navigate('/dashboard');
      } else {
        setError(result.error);
      }
    } catch (err) {
      setError('Login failed. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="login-container">
      <div className="login-content">
        <div className="text-center mb-4">
          <h1 className="app-title">POS System</h1>
          <p className="app-subtitle">Point of Sale Management Software</p>
        </div>
          
          <Card className="login-card">
            <Card.Body>
              <div className="text-center mb-4">
                <div className="login-icon">
                  <FaUser size={40} />
                </div>
                <h2>Login</h2>
                <p className="text-muted">Enter your credentials to access the system</p>
              </div>
              
              {error && <Alert variant="danger">{error}</Alert>}
              
              <Form onSubmit={handleSubmit}>
                <Form.Group className="mb-3">
                  <Form.Label>Email</Form.Label>
                  <div className="input-group">
                    <span className="input-group-text">
                      <FaUser />
                    </span>
                    <Form.Control
                      type="email"
                      name="email"
                      value={credentials.email}
                      onChange={handleChange}
                      placeholder="Enter email"
                      required
                    />
                  </div>
                </Form.Group>
                
                <Form.Group className="mb-4">
                  <Form.Label>Password</Form.Label>
                  <div className="input-group">
                    <span className="input-group-text">
                      <FaLock />
                    </span>
                    <Form.Control
                      type="password"
                      name="password"
                      value={credentials.password}
                      onChange={handleChange}
                      placeholder="Enter password"
                      required
                    />
                  </div>
                </Form.Group>
                
                <Button 
                  variant="primary" 
                  type="submit" 
                  className="w-100" 
                  disabled={loading}
                >
                  {loading ? 'Logging in...' : (
                    <>
                      <FaSignInAlt className="me-2" /> Login
                    </>
                  )}
                </Button>
              </Form>
              
              <div className="text-center mt-3">
                <p className="text-muted small">
                  Demo credentials: admin@example.com / admin123
                </p>
                <Button 
                  variant="outline-secondary" 
                  size="sm"
                  onClick={async () => {
                    console.log('Testing API directly...');
                    try {
                      const response = await fetch('http://localhost:5000/api/auth/test');
                      const data = await response.json();
                      console.log('API Test Success:', data);
                      alert('API Test Success: ' + JSON.stringify(data));
                    } catch (error) {
                      console.error('API Test Error:', error);
                      alert('API Test Error: ' + error.message);
                    }
                  }}
                >
                  Test API Connection
                </Button>
              </div>
            </Card.Body>
          </Card>
          
          <div className="text-center mt-3">
            <p className="text-muted small">
              &copy; {new Date().getFullYear()} POS System. All rights reserved.
            </p>
          </div>
        </div>
    </div>
  );
};

export default Login;