import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Form, Button, Table, Badge, Spinner, Alert, Dropdown, ButtonGroup } from 'react-bootstrap';
import { FaMobileAlt, FaDownload, FaCalendarAlt, FaChartLine, FaEye, FaFileExport, FaSearch } from 'react-icons/fa';
import axios from 'axios';
import { useAuth } from '../../contexts/AuthContext';
import { useNavigate } from 'react-router-dom';
import './BkashAccountSelection.css';

const BkashAccountSelection = () => {
  const { currentUser } = useAuth();
  const navigate = useNavigate();
  const [alert, setAlert] = useState({ show: false, message: '', variant: 'success' });
  const [loading, setLoading] = useState(false);
  const [accountNumbers, setAccountNumbers] = useState([]);
  const [selectedAccount, setSelectedAccount] = useState(null);
  const [accountData, setAccountData] = useState(null);
  const [accountSummary, setAccountSummary] = useState(null);
  const [transactions, setTransactions] = useState([]);
  const [viewType, setViewType] = useState('daily'); // daily, weekly, monthly
  const [startDate, setStartDate] = useState('');
  const [endDate, setEndDate] = useState('');
  const [transactionLoading, setTransactionLoading] = useState(false);
  const [openingBalance, setOpeningBalance] = useState('');
  const [closingBalance, setClosingBalance] = useState('');
  const [balanceDate, setBalanceDate] = useState(new Date().toISOString().split('T')[0]);
  const [savingBalance, setSavingBalance] = useState(false);
  const [dateRange, setDateRange] = useState({
    startDate: new Date().toISOString().split('T')[0],
    endDate: new Date().toISOString().split('T')[0]
  });

  const showAlert = (message, variant = 'success') => {
    setAlert({ show: true, message, variant });
    setTimeout(() => setAlert({ show: false, message: '', variant: 'success' }), 5000);
  };

  useEffect(() => {
    if (currentUser) {
      fetchAccountNumbers();
    }
  }, [currentUser]);

  // Initialize date range when component mounts
  useEffect(() => {
    const dateRange = getDateRangeForView('daily');
    setStartDate(dateRange.start);
    setEndDate(dateRange.end);
  }, []);

  // Fetch transaction history when account or date filters change
  useEffect(() => {
    if (selectedAccount && startDate && endDate) {
      fetchTransactionHistory();
    }
  }, [selectedAccount, startDate, endDate]);

  useEffect(() => {
    if (selectedAccount) {
      fetchAccountData();
    }
  }, [selectedAccount, dateRange]);

  const fetchAccountNumbers = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-account-numbers', {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      // Filter only active accounts
      const activeAccounts = response.data.filter(account => account.isActive);
      setAccountNumbers(activeAccounts);
      
      if (activeAccounts.length > 0) {
        // Auto-select the first primary account or first account
        const primaryAccount = activeAccounts.find(acc => acc.isPrimary) || activeAccounts[0];
        setSelectedAccount(primaryAccount);
      }
    } catch (error) {
      console.error('Error fetching account numbers:', error);
      showAlert('Error fetching account numbers', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchAccountData = async () => {
    if (!selectedAccount) return;
    
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      // Fetch account data based on account type
      let accountResponse;
      let transactionResponse;
      
      switch (selectedAccount.accountType) {
        case 'agent':
          accountResponse = await axios.get(`http://localhost:5000/api/bkash-agent/${dateRange.startDate}`, {
            headers: { Authorization: `Bearer ${token}` }
          });
          break;
        case 'personal':
          accountResponse = await axios.get(`http://localhost:5000/api/bkash-personal/${dateRange.startDate}`, {
            headers: { Authorization: `Bearer ${token}` }
          });
          break;
        case 'merchant':
          accountResponse = await axios.get(`http://localhost:5000/api/bkash-merchant/${dateRange.startDate}`, {
            headers: { Authorization: `Bearer ${token}` }
          });
          break;
        default:
          throw new Error('Invalid account type');
      }
      
      setAccountData(accountResponse.data);
      
      // Fetch transactions for the selected date range
      const transactionParams = new URLSearchParams({
        accountType: selectedAccount.accountType,
        accountNumberId: selectedAccount.id,
        startDate: dateRange.startDate,
        endDate: dateRange.endDate
      });
      
      transactionResponse = await axios.get(`http://localhost:5000/api/bkash-transactions?${transactionParams}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      // Fetch account summary
      const summaryParams = new URLSearchParams({
        accountType: selectedAccount.accountType,
        accountNumberId: selectedAccount.id,
        startDate: dateRange.startDate,
        endDate: dateRange.endDate
      });
      
      const summaryResponse = await axios.get(`http://localhost:5000/api/bkash-transactions/account-summary?${summaryParams}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      setAccountSummary(summaryResponse.data);
      
      setTransactions(transactionResponse.data || []);
      
    } catch (error) {
      console.error('Error fetching account data:', error);
      if (error.response?.status === 404) {
        // No data for this date, create empty structure
        setAccountData(createEmptyAccountData(selectedAccount.accountType));
        setTransactions([]);
      } else {
        showAlert('Error fetching account data', 'danger');
      }
    } finally {
      setLoading(false);
    }
  };

  const fetchTransactionHistory = async () => {
    if (!selectedAccount) return;
    
    setTransactionLoading(true);
    try {
      const token = localStorage.getItem('token');
      const params = new URLSearchParams({
        accountType: selectedAccount.accountType,
        accountNumberId: selectedAccount.id.toString()
      });

      // Add date filters if provided
      if (startDate) params.append('startDate', startDate);
      if (endDate) params.append('endDate', endDate);

      const response = await axios.get(`http://localhost:5000/api/bkash-transactions/transactions?${params}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTransactions(response.data.transactions || []);
    } catch (error) {
      console.error('Error fetching transaction history:', error);
      setTransactions([]);
    } finally {
      setTransactionLoading(false);
    }
  };

  const getDateRangeForView = (type) => {
    const today = new Date();
    let start, end;

    switch (type) {
      case 'daily':
        start = new Date(today);
        end = new Date(today);
        break;
      case 'weekly':
        start = new Date(today);
        start.setDate(today.getDate() - 7);
        end = new Date(today);
        break;
      case 'monthly':
        start = new Date(today);
        start.setDate(1); // First day of current month
        end = new Date(today);
        break;
      default:
        start = new Date(today);
        end = new Date(today);
    }

    return {
      start: start.toISOString().split('T')[0],
      end: end.toISOString().split('T')[0]
    };
  };

  const handleViewTypeChange = (type) => {
    setViewType(type);
    const dateRange = getDateRangeForView(type);
    setStartDate(dateRange.start);
    setEndDate(dateRange.end);
  };

  const saveOpeningBalance = async () => {
    if (!selectedAccount || !openingBalance || !balanceDate) {
      showAlert('Please select an account, enter opening balance, and select a date', 'warning');
      return;
    }

    setSavingBalance(true);
    try {
      const token = localStorage.getItem('token');
      const response = await axios.post('http://localhost:5000/api/bkash-transactions/save-balance', {
        accountType: selectedAccount.accountType,
        accountNumberId: selectedAccount.id,
        balanceType: 'opening',
        amount: parseFloat(openingBalance),
        date: balanceDate
      }, {
        headers: { Authorization: `Bearer ${token}` }
      });

      if (response.data.success) {
        showAlert('Opening balance saved successfully!', 'success');
        setOpeningBalance('');
        // Refresh account data to show updated balance
        fetchAccountData();
      }
    } catch (error) {
      console.error('Error saving opening balance:', error);
      showAlert('Error saving opening balance. Please try again.', 'danger');
    } finally {
      setSavingBalance(false);
    }
  };

  const saveClosingBalance = async () => {
    if (!selectedAccount || !closingBalance || !balanceDate) {
      showAlert('Please select an account, enter closing balance, and select a date', 'warning');
      return;
    }

    setSavingBalance(true);
    try {
      const token = localStorage.getItem('token');
      const response = await axios.post('http://localhost:5000/api/bkash-transactions/save-balance', {
        accountType: selectedAccount.accountType,
        accountNumberId: selectedAccount.id,
        balanceType: 'closing',
        amount: parseFloat(closingBalance),
        date: balanceDate
      }, {
        headers: { Authorization: `Bearer ${token}` }
      });

      if (response.data.success) {
        showAlert('Closing balance saved successfully!', 'success');
        setClosingBalance('');
        // Refresh account data to show updated balance
        fetchAccountData();
      }
    } catch (error) {
      console.error('Error saving closing balance:', error);
      showAlert('Error saving closing balance. Please try again.', 'danger');
    } finally {
      setSavingBalance(false);
    }
  };

  const createEmptyAccountData = (accountType) => {
    const baseData = {
      date: dateRange.startDate,
      openingBalance: 0,
      closingBalance: null,
      notes: '',
      isCompleted: false
    };

    switch (accountType) {
      case 'agent':
        return {
          ...baseData,
          srReceived: 0,
          srGiven: 0,
          bankDeposit: 0,
          bankWithdraw: 0,
          dailyProfit: null
        };
      case 'personal':
        return {
          ...baseData,
          totalSent: 0,
          totalReceived: 0,
          netChange: null
        };
      case 'merchant':
        return {
          ...baseData,
          totalSales: 0,
          totalRefunds: 0,
          totalWithdrawals: 0,
          merchantFee: 0,
          netRevenue: null,
          netChange: null
        };
      default:
        return baseData;
    }
  };

  const calculateDailyProfit = (data, accountType) => {
    if (!data) return 0;
    
    switch (accountType) {
      case 'agent':
        if (data.closingBalance !== null) {
          return parseFloat(data.closingBalance) - parseFloat(data.openingBalance) + 
                 parseFloat(data.srReceived || 0) - parseFloat(data.srGiven || 0);
        }
        return null;
      case 'personal':
        return data.netChange;
      case 'merchant':
        return data.netRevenue;
      default:
        return 0;
    }
  };

  const groupAccountsByType = () => {
    const grouped = {
      agent: accountNumbers.filter(acc => acc.accountType === 'agent'),
      personal: accountNumbers.filter(acc => acc.accountType === 'personal'),
      merchant: accountNumbers.filter(acc => acc.accountType === 'merchant')
    };
    return grouped;
  };

  const getAccountTypeLabel = (type) => {
    switch (type) {
      case 'agent': return 'Agent';
      case 'personal': return 'Personal';
      case 'merchant': return 'Merchant';
      default: return type;
    }
  };

  const getAccountTypeBadge = (type) => {
    switch (type) {
      case 'agent': return 'primary';
      case 'personal': return 'success';
      case 'merchant': return 'info';
      default: return 'secondary';
    }
  };

  const handleExportData = async (format) => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const exportData = {
        account: selectedAccount,
        accountData: accountData,
        transactions: transactions,
        dateRange: dateRange,
        dailyProfit: calculateDailyProfit(accountData, selectedAccount.accountType)
      };
      
      const response = await axios.post(`http://localhost:5000/api/bkash-reports/export/${format}`, exportData, {
        headers: { Authorization: `Bearer ${token}` },
        responseType: 'blob'
      });
      
      // Create download link
      const url = window.URL.createObjectURL(new Blob([response.data]));
      const link = document.createElement('a');
      link.href = url;
      link.setAttribute('download', `bkash-${selectedAccount.accountNumber}-${dateRange.startDate}.${format}`);
      document.body.appendChild(link);
      link.click();
      link.remove();
      
      showAlert(`Data exported successfully as ${format.toUpperCase()}`, 'success');
    } catch (error) {
      console.error('Error exporting data:', error);
      showAlert('Error exporting data', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const navigateToReports = () => {
    if (selectedAccount) {
      navigate(`/bkash/reports?accountId=${selectedAccount.id}&accountType=${selectedAccount.accountType}`);
    }
  };

  const navigateToAllAccountsReport = () => {
    navigate('/bkash/reports/all-accounts');
  };

  const groupedAccounts = groupAccountsByType();
  const dailyProfit = calculateDailyProfit(accountData, selectedAccount?.accountType);

  return (
    <Container fluid className="bkash-account-selection">
      {/* Alert */}
      {alert.show && (
        <Alert variant={alert.variant} dismissible onClose={() => setAlert({ show: false, message: '', variant: 'success' })}>
          {alert.message}
        </Alert>
      )}

      {/* Header */}
      <div className="page-header">
        <div className="d-flex justify-content-between align-items-center">
          <div>
            <h4 className="page-title">
              <FaMobileAlt className="me-2" />
              Bkash Account Management
            </h4>
            <p className="page-subtitle">Select an account to view details and manage transactions</p>
          </div>
          <div className="header-actions">
            <Button 
              variant="outline-info"
              onClick={navigateToAllAccountsReport}
              className="me-2"
            >
              <FaChartLine className="me-2" />
              All Accounts Report
            </Button>
            <Button 
              variant="outline-primary"
              onClick={() => navigate('/bkash/account-numbers')}
            >
              <FaEye className="me-2" />
              Manage Accounts
            </Button>
          </div>
        </div>
      </div>

      {/* Account Selection */}
      <Row className="mb-4">
        <Col md={6}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">Select Bkash Account</h5>
            </Card.Header>
            <Card.Body>
              <Form.Group>
                <Form.Label>Account Number</Form.Label>
                <Form.Select
                  value={selectedAccount?.id || ''}
                  onChange={(e) => {
                    const accountId = parseInt(e.target.value);
                    const account = accountNumbers.find(acc => acc.id === accountId);
                    setSelectedAccount(account);
                  }}
                  disabled={loading}
                >
                  <option value="">Select an account...</option>
                  
                  {/* Agent Accounts */}
                  {groupedAccounts.agent.length > 0 && (
                    <optgroup label="Agent Accounts">
                      {groupedAccounts.agent.map(account => (
                        <option key={account.id} value={account.id}>
                          {account.accountNumber} - {account.accountName || 'Agent Account'}
                          {account.isPrimary ? ' (Primary)' : ''}
                        </option>
                      ))}
                    </optgroup>
                  )}
                  
                  {/* Personal Accounts */}
                  {groupedAccounts.personal.length > 0 && (
                    <optgroup label="Personal Accounts">
                      {groupedAccounts.personal.map(account => (
                        <option key={account.id} value={account.id}>
                          {account.accountNumber} - {account.accountName || 'Personal Account'}
                          {account.isPrimary ? ' (Primary)' : ''}
                        </option>
                      ))}
                    </optgroup>
                  )}
                  
                  {/* Merchant Accounts */}
                  {groupedAccounts.merchant.length > 0 && (
                    <optgroup label="Merchant Accounts">
                      {groupedAccounts.merchant.map(account => (
                        <option key={account.id} value={account.id}>
                          {account.accountNumber} - {account.accountName || 'Merchant Account'}
                          {account.isPrimary ? ' (Primary)' : ''}
                        </option>
                      ))}
                    </optgroup>
                  )}
                </Form.Select>
              </Form.Group>
            </Card.Body>
          </Card>
        </Col>
        
        <Col md={6}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">Date Range</h5>
            </Card.Header>
            <Card.Body>
              <Row>
                <Col md={6}>
                  <Form.Group>
                    <Form.Label>Start Date</Form.Label>
                    <Form.Control
                      type="date"
                      value={dateRange.startDate}
                      onChange={(e) => setDateRange(prev => ({ ...prev, startDate: e.target.value }))}
                      disabled={loading}
                    />
                  </Form.Group>
                </Col>
                <Col md={6}>
                  <Form.Group>
                    <Form.Label>End Date</Form.Label>
                    <Form.Control
                      type="date"
                      value={dateRange.endDate}
                      onChange={(e) => setDateRange(prev => ({ ...prev, endDate: e.target.value }))}
                      disabled={loading}
                    />
                  </Form.Group>
                </Col>
              </Row>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Account Details */}
      {selectedAccount && (
        <>
          {/* Account Info Card */}
          <Row className="mb-4">
            <Col md={12}>
              <Card className="account-info-card">
                <Card.Header className="d-flex justify-content-between align-items-center">
                  <div>
                    <h5 className="mb-0">
                      <Badge bg={getAccountTypeBadge(selectedAccount.accountType)} className="me-2">
                        {getAccountTypeLabel(selectedAccount.accountType)}
                      </Badge>
                      {selectedAccount.accountNumber}
                    </h5>
                    <small className="text-muted">{selectedAccount.accountName}</small>
                  </div>
                  <div className="d-flex gap-2">
                    <Button
                      variant="outline-success"
                      size="sm"
                      onClick={navigateToReports}
                    >
                      <FaChartLine className="me-1" />
                      View Reports
                    </Button>
                    <Dropdown>
                      <Dropdown.Toggle variant="outline-primary" size="sm">
                        <FaFileExport className="me-1" />
                        Export
                      </Dropdown.Toggle>
                      <Dropdown.Menu>
                        <Dropdown.Item onClick={() => handleExportData('pdf')}>
                          Export as PDF
                        </Dropdown.Item>
                        <Dropdown.Item onClick={() => handleExportData('excel')}>
                          Export as Excel
                        </Dropdown.Item>
                      </Dropdown.Menu>
                    </Dropdown>
                  </div>
                </Card.Header>
                <Card.Body>
                  {loading ? (
                    <div className="text-center py-4">
                      <Spinner animation="border" />
                      <p className="mt-2">Loading account data...</p>
                    </div>
                  ) : accountSummary ? (
                    <Row>
                      <Col md={3}>
                        <div className="balance-card opening">
                          <h6>Opening Balance</h6>
                          <h4>৳{parseFloat(accountSummary.openingBalance || 0).toLocaleString()}</h4>
                        </div>
                      </Col>
                      <Col md={3}>
                        <div className="balance-card closing">
                          <h6>Closing Balance</h6>
                          <h4>৳{parseFloat(accountSummary.closingBalance || 0).toLocaleString()}</h4>
                        </div>
                      </Col>
                      <Col md={3}>
                        <div className="balance-card received">
                          <h6>Amount Received</h6>
                          <h4>৳{parseFloat(accountSummary.totalReceived || 0).toLocaleString()}</h4>
                        </div>
                      </Col>
                      <Col md={3}>
                        <div className="balance-card profit">
                          <h6>Daily Profit</h6>
                          <h4 className={parseFloat(accountSummary.dailyProfit || 0) >= 0 ? 'text-success' : 'text-danger'}>
                            ৳{parseFloat(accountSummary.dailyProfit || 0).toLocaleString()}
                          </h4>
                        </div>
                      </Col>
                    </Row>
                  ) : (
                    <div className="text-center py-4">
                      <p className="text-muted">No data available for the selected date range</p>
                    </div>
                  )}
                </Card.Body>
              </Card>
            </Col>
          </Row>

          {/* Balance Input Section */}
          <Row className="mb-4">
            <Col md={12}>
              <Card className="balance-management-card">
                <Card.Header>
                  <h5 className="mb-0">Balance Management</h5>
                  <small className="text-muted">Add opening balance at start of day or closing balance at end of day</small>
                </Card.Header>
                <Card.Body>
                  <Row>
                    <Col md={4}>
                      <Form.Group className="mb-3">
                        <Form.Label>Date <span className="text-danger">*</span></Form.Label>
                        <Form.Control
                          type="date"
                          value={balanceDate}
                          onChange={(e) => setBalanceDate(e.target.value)}
                          max={new Date().toISOString().split('T')[0]}
                        />
                      </Form.Group>
                    </Col>
                  </Row>
                  
                  <Row>
                    {/* Opening Balance Input */}
                    <Col md={6}>
                      <Card className="balance-input-card h-100">
                        <Card.Header className="bg-light">
                          <h6 className="mb-0 text-primary">Opening Balance</h6>
                          <small className="text-muted">Set the starting balance for the day</small>
                        </Card.Header>
                        <Card.Body>
                          <Form.Group className="mb-3">
                            <Form.Label>Amount (৳) <span className="text-danger">*</span></Form.Label>
                            <Form.Control
                              type="number"
                              placeholder="Enter opening balance amount"
                              value={openingBalance}
                              onChange={(e) => setOpeningBalance(e.target.value)}
                              min="0"
                              step="0.01"
                            />
                          </Form.Group>
                          <Button 
                            variant="primary" 
                            onClick={saveOpeningBalance}
                            disabled={savingBalance || !openingBalance || !balanceDate}
                            className="w-100"
                          >
                            {savingBalance ? (
                              <>
                                <Spinner animation="border" size="sm" className="me-2" />
                                Saving...
                              </>
                            ) : (
                              'Save Opening Balance'
                            )}
                          </Button>
                        </Card.Body>
                      </Card>
                    </Col>

                    {/* Closing Balance Input */}
                    <Col md={6}>
                      <Card className="balance-input-card h-100">
                        <Card.Header className="bg-light">
                          <h6 className="mb-0 text-success">Closing Balance</h6>
                          <small className="text-muted">Set the ending balance for the day</small>
                        </Card.Header>
                        <Card.Body>
                          <Form.Group className="mb-3">
                            <Form.Label>Amount (৳) <span className="text-danger">*</span></Form.Label>
                            <Form.Control
                              type="number"
                              placeholder="Enter closing balance amount"
                              value={closingBalance}
                              onChange={(e) => setClosingBalance(e.target.value)}
                              min="0"
                              step="0.01"
                            />
                          </Form.Group>
                          <Button 
                            variant="success" 
                            onClick={saveClosingBalance}
                            disabled={savingBalance || !closingBalance || !balanceDate}
                            className="w-100"
                          >
                            {savingBalance ? (
                              <>
                                <Spinner animation="border" size="sm" className="me-2" />
                                Saving...
                              </>
                            ) : (
                              'Save Closing Balance'
                            )}
                          </Button>
                        </Card.Body>
                      </Card>
                    </Col>
                  </Row>
                </Card.Body>
              </Card>
            </Col>
          </Row>

          {/* Account Type Specific Details */}
          {accountData && (
            <Row className="mb-4">
              <Col md={12}>
                <Card>
                  <Card.Header>
                    <h5 className="mb-0">Transaction Details</h5>
                  </Card.Header>
                  <Card.Body>
                    {selectedAccount.accountType === 'agent' && (
                      <Row>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Amount from SR</h6>
                            <h5 className="text-success">৳{parseFloat(accountData.srReceived || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Amount to SR</h6>
                            <h5 className="text-danger">৳{parseFloat(accountData.srGiven || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Bank Deposit</h6>
                            <h5 className="text-info">৳{parseFloat(accountData.bankDeposit || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Bank Withdraw</h6>
                            <h5 className="text-warning">৳{parseFloat(accountData.bankWithdraw || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                      </Row>
                    )}

                    {selectedAccount.accountType === 'personal' && (
                      <Row>
                        <Col md={6}>
                          <div className="detail-card">
                            <h6>Total Sent</h6>
                            <h5 className="text-danger">৳{parseFloat(accountData.totalSent || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={6}>
                          <div className="detail-card">
                            <h6>Total Received</h6>
                            <h5 className="text-success">৳{parseFloat(accountData.totalReceived || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                      </Row>
                    )}

                    {selectedAccount.accountType === 'merchant' && (
                      <Row>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Total Sales</h6>
                            <h5 className="text-success">৳{parseFloat(accountData.totalSales || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Total Refunds</h6>
                            <h5 className="text-danger">৳{parseFloat(accountData.totalRefunds || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Withdrawals</h6>
                            <h5 className="text-warning">৳{parseFloat(accountData.totalWithdrawals || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                        <Col md={3}>
                          <div className="detail-card">
                            <h6>Merchant Fee</h6>
                            <h5 className="text-info">৳{parseFloat(accountData.merchantFee || 0).toLocaleString()}</h5>
                          </div>
                        </Col>
                      </Row>
                    )}
                  </Card.Body>
                </Card>
              </Col>
            </Row>
          )}

          {/* Transaction History */}
          <Row className="mb-4">
            <Col md={12}>
              <Card>
                <Card.Header className="d-flex justify-content-between align-items-center">
                  <h5 className="mb-0">Transaction History</h5>
                  <div className="d-flex align-items-center gap-3">
                    {/* View Type Buttons */}
                    <ButtonGroup size="sm">
                      <Button 
                        variant={viewType === 'daily' ? 'primary' : 'outline-primary'}
                        onClick={() => handleViewTypeChange('daily')}
                      >
                        Daily
                      </Button>
                      <Button 
                        variant={viewType === 'weekly' ? 'primary' : 'outline-primary'}
                        onClick={() => handleViewTypeChange('weekly')}
                      >
                        Weekly
                      </Button>
                      <Button 
                        variant={viewType === 'monthly' ? 'primary' : 'outline-primary'}
                        onClick={() => handleViewTypeChange('monthly')}
                      >
                        Monthly
                      </Button>
                    </ButtonGroup>
                    
                    {/* Custom Date Range */}
                    <div className="d-flex align-items-center gap-2">
                      <Form.Control
                        type="date"
                        size="sm"
                        value={startDate}
                        onChange={(e) => setStartDate(e.target.value)}
                        style={{ width: '140px' }}
                      />
                      <span className="text-muted">to</span>
                      <Form.Control
                        type="date"
                        size="sm"
                        value={endDate}
                        onChange={(e) => setEndDate(e.target.value)}
                        style={{ width: '140px' }}
                      />
                      <Button 
                        size="sm" 
                        variant="outline-secondary"
                        onClick={fetchTransactionHistory}
                        disabled={transactionLoading}
                      >
                        <FaSearch />
                      </Button>
                    </div>
                  </div>
                </Card.Header>
                <Card.Body>
                  {transactionLoading ? (
                    <div className="text-center py-4">
                      <Spinner animation="border" />
                      <p className="mt-2">Loading transactions...</p>
                    </div>
                  ) : transactions.length > 0 ? (
                    <div className="table-responsive">
                      <Table striped hover>
                        <thead>
                          <tr>
                            <th>Date</th>
                            <th>Time</th>
                            <th>Type</th>
                            <th>Amount</th>
                            <th>Description</th>
                            <th>Reference</th>
                          </tr>
                        </thead>
                        <tbody>
                          {transactions.map((transaction, index) => (
                            <tr key={index}>
                              <td>{new Date(transaction.date).toLocaleDateString()}</td>
                              <td>{transaction.time}</td>
                              <td>
                                <Badge bg="secondary">
                                  {transaction.type.replace('_', ' ').toUpperCase()}
                                </Badge>
                              </td>
                              <td className={transaction.amount > 0 ? 'text-success' : 'text-danger'}>
                                ৳{parseFloat(transaction.amount).toLocaleString()}
                              </td>
                              <td>{transaction.description || '-'}</td>
                              <td>{transaction.reference || '-'}</td>
                            </tr>
                          ))}
                        </tbody>
                      </Table>
                    </div>
                  ) : (
                    <div className="text-center py-4">
                      <FaCalendarAlt className="fa-3x text-muted mb-3" />
                      <h5>No Transactions</h5>
                      <p className="text-muted">No transactions found for the selected date range</p>
                    </div>
                  )}
                </Card.Body>
              </Card>
            </Col>
          </Row>
        </>
      )}

      {/* No Account Selected */}
      {!selectedAccount && !loading && (
        <Row>
          <Col md={12}>
            <Card>
              <Card.Body>
                <div className="text-center py-5">
                  <FaMobileAlt className="fa-3x text-muted mb-3" />
                  <h5>Select a Bkash Account</h5>
                  <p className="text-muted">Choose an account from the dropdown above to view details and transactions</p>
                </div>
              </Card.Body>
            </Card>
          </Col>
        </Row>
      )}
    </Container>
  );
};

export default BkashAccountSelection;