import React, { useState, useEffect } from 'react';
import { Row, Col, Card, Form, Button, Table, Badge, Spinner, Modal, InputGroup } from 'react-bootstrap';
import { FaPlus, FaList } from 'react-icons/fa';
import axios from 'axios';
import { useAuth } from '../../contexts/AuthContext';

const BkashAgentAccount = ({ showAlert }) => {
  const { currentUser } = useAuth();
  const [accountData, setAccountData] = useState(null);
  const [loading, setLoading] = useState(false);
  const [agentNumbers, setAgentNumbers] = useState([]);
  const [selectedAgent, setSelectedAgent] = useState(null);
  const [showAgentModal, setShowAgentModal] = useState(false);
  const [formData, setFormData] = useState({
    openingBalance: '',
    closingBalance: '',
    srReceived: '',
    srGiven: '',
    b2bSend: '',
    b2bReceived: '',
    amountReceived: '',
    bkashAccountNumberId: '',
    commissionRate: '1.8',
    totalTransaction: '',
    totalCommission: '',
    date: new Date().toISOString().split('T')[0]
  });
  const [showCloseModal, setShowCloseModal] = useState(false);
  const [summary, setSummary] = useState(null);
  
  // SR Transaction Modal States
  const [showSRReceivedModal, setShowSRReceivedModal] = useState(false);
  const [showSRGivenModal, setShowSRGivenModal] = useState(false);
  const [showSRListModal, setShowSRListModal] = useState(false);
  const [srTransactionType, setSRTransactionType] = useState('');
  
  // SR Transaction Form Data
  const [srTransaction, setSRTransaction] = useState({
    amount: '',
    description: '',
    reference: '',
    notes: ''
  });
  
  // SR Transactions List
  const [srTransactions, setSRTransactions] = useState([]);

  useEffect(() => {
    // Only fetch data if user is authenticated
    if (currentUser) {
      fetchAgentNumbers();
      fetchSummary();
    }
  }, [currentUser]);

  useEffect(() => {
    // Fetch account data when agent is selected
    if (selectedAgent) {
      fetchTodayAccount();
    }
  }, [selectedAgent]);

  const fetchAgentNumbers = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-account-numbers?accountType=agent&isActive=true', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAgentNumbers(response.data);
      
      // Auto-select primary agent if available
      const primaryAgent = response.data.find(agent => agent.isPrimary);
      if (primaryAgent && !selectedAgent) {
        setSelectedAgent(primaryAgent);
        setFormData(prev => ({ ...prev, bkashAccountNumberId: primaryAgent.id }));
      }
    } catch (error) {
      console.error('Error fetching agent numbers:', error);
      showAlert('Error fetching agent numbers', 'danger');
    }
  };

  const fetchTodayAccount = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-agent/today', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAccountData(response.data);
      if (response.data) {
        setFormData({
          openingBalance: response.data.openingBalance || '',
          closingBalance: response.data.closingBalance || '',
          srReceived: response.data.srReceived || '',
          srGiven: response.data.srGiven || '',
          b2bSend: response.data.b2bSend || '',
          b2bReceived: response.data.b2bReceived || '',
          amountReceived: response.data.amountReceived || '',
          bkashAccountNumberId: response.data.bkashAccountNumberId || selectedAgent?.id || '',
          commissionRate: response.data.commissionRate || '1.8',
          totalTransaction: response.data.totalTransaction || '',
          totalCommission: response.data.totalCommission || '',
          date: response.data.date || new Date().toISOString().split('T')[0]
        });
      }
    } catch (error) {
      console.error('Error fetching today account:', error);
      showAlert('Error fetching account data', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchSummary = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-agent/summary', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setSummary(response.data);
    } catch (error) {
      console.error('Error fetching summary:', error);
    }
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  // Handle SR Transaction form changes
  const handleSRTransactionChange = (e) => {
    const { name, value } = e.target;
    setSRTransaction(prev => ({
      ...prev,
      [name]: value
    }));
  };

  // Open SR Transaction Modals
  const openSRReceivedModal = () => {
    setSRTransactionType('received');
    setShowSRReceivedModal(true);
  };

  const openSRGivenModal = () => {
    setSRTransactionType('given');
    setShowSRGivenModal(true);
  };

  // Close SR Transaction Modals
  const closeSRModals = () => {
    setShowSRReceivedModal(false);
    setShowSRGivenModal(false);
    setSRTransaction({
      amount: '',
      description: '',
      reference: '',
      notes: ''
    });
  };

  // Add SR Transaction
  const addSRTransaction = () => {
    if (!srTransaction.amount || parseFloat(srTransaction.amount) <= 0) {
      showAlert('Please enter a valid amount', 'danger');
      return;
    }

    const newTransaction = {
      id: Date.now(),
      type: srTransactionType,
      amount: parseFloat(srTransaction.amount),
      description: srTransaction.description || `SR ${srTransactionType}`,
      reference: srTransaction.reference,
      notes: srTransaction.notes,
      date: new Date().toISOString(),
      timestamp: new Date().toLocaleString()
    };

    setSRTransactions(prev => [...prev, newTransaction]);

    // Update the main form data
    const currentAmount = parseFloat(formData[srTransactionType === 'received' ? 'srReceived' : 'srGiven']) || 0;
    const newAmount = currentAmount + parseFloat(srTransaction.amount);
    
    setFormData(prev => ({
      ...prev,
      [srTransactionType === 'received' ? 'srReceived' : 'srGiven']: newAmount.toString()
    }));

    showAlert(`SR ${srTransactionType} transaction added successfully!`, 'success');
    closeSRModals();
  };

  // Remove SR Transaction
  const removeSRTransaction = (transactionId) => {
    const transaction = srTransactions.find(t => t.id === transactionId);
    if (transaction) {
      setSRTransactions(prev => prev.filter(t => t.id !== transactionId));
      
      // Update the main form data
      const currentAmount = parseFloat(formData[transaction.type === 'received' ? 'srReceived' : 'srGiven']) || 0;
      const newAmount = Math.max(0, currentAmount - transaction.amount);
      
      setFormData(prev => ({
        ...prev,
        [transaction.type === 'received' ? 'srReceived' : 'srGiven']: newAmount.toString()
      }));

      showAlert(`SR ${transaction.type} transaction removed!`, 'info');
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...formData,
        openingBalance: parseFloat(formData.openingBalance) || 0,
        closingBalance: parseFloat(formData.closingBalance) || 0,
        srReceived: parseFloat(formData.srReceived) || 0,
        srGiven: parseFloat(formData.srGiven) || 0,
        commissionRate: parseFloat(formData.commissionRate) || 1.8
      };

      await axios.post('http://localhost:5000/api/bkash-agent', submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Agent account updated successfully!', 'success');
      fetchTodayAccount();
      fetchSummary();
    } catch (error) {
      console.error('Error updating account:', error);
      showAlert('Error updating account', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleCloseDay = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      // Calculate the next day's opening balance including B2B transactions
      const currentClosing = parseFloat(formData.closingBalance) || 0;
      const b2bSend = parseFloat(formData.b2bSend) || 0;
      const b2bReceived = parseFloat(formData.b2bReceived) || 0;
      
      // Next day opening = Current closing + B2B Received - B2B Send
      const nextDayOpening = currentClosing + b2bReceived - b2bSend;
      
      await axios.put('http://localhost:5000/api/bkash-agent/close', {
        date: formData.date,
        closingBalance: currentClosing,
        b2bSend: b2bSend,
        b2bReceived: b2bReceived,
        nextDayOpening: nextDayOpening,
        bkashAccountNumberId: formData.bkashAccountNumberId
      }, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert(`Day closed successfully! Next day opening balance: ৳${nextDayOpening.toLocaleString()}`, 'success');
      setShowCloseModal(false);
      fetchTodayAccount();
      fetchSummary();
    } catch (error) {
      console.error('Error closing day:', error);
      showAlert('Error closing day', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const calculateProfit = () => {
    if (!accountData) return 0;
    const opening = parseFloat(accountData.openingBalance) || 0;
    const closing = parseFloat(formData.closingBalance) || parseFloat(accountData.closingBalance) || 0;
    const b2bSend = parseFloat(formData.b2bSend) || parseFloat(accountData.b2bSend) || 0;
    const b2bReceived = parseFloat(formData.b2bReceived) || parseFloat(accountData.b2bReceived) || 0;
    
    // Profit = Closing - Opening (B2B transactions are already factored into the balance)
    // But we show the actual profit excluding B2B transfers
    const baseProfit = closing - opening;
    return baseProfit;
  };

  if (loading && !accountData) {
    return (
      <div className="loading-spinner">
        <Spinner animation="border" variant="primary" />
      </div>
    );
  }

  return (
    <div>
      {/* Agent Selection */}
      <Row className="mb-4">
        <Col>
          <Card>
            <Card.Body>
              <div className="d-flex justify-content-between align-items-center">
                <h5 className="mb-0">
                  <i className="fas fa-mobile-alt me-2"></i>
                  bKash Agent Account
                </h5>
                <div className="d-flex align-items-center">
                  {selectedAgent ? (
                    <div className="me-3">
                      <Badge bg="success" className="me-2">Selected Agent</Badge>
                      <strong>{selectedAgent.accountNumber}</strong> - {selectedAgent.accountName}
                    </div>
                  ) : (
                    <div className="me-3">
                      <Badge bg="warning">No Agent Selected</Badge>
                    </div>
                  )}
                  <Button 
                    variant="outline-primary" 
                    size="sm"
                    onClick={() => setShowAgentModal(true)}
                  >
                    {selectedAgent ? 'Change Agent' : 'Select Agent'}
                  </Button>
                </div>
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Summary Cards */}
      {summary && selectedAgent && (
        <Row className="mb-4">
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">৳{summary.totalProfit?.toLocaleString() || '0'}</div>
                <div className="summary-label">Total Profit</div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">{summary.totalDays || '0'}</div>
                <div className="summary-label">Active Days</div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">৳{summary.avgProfit?.toLocaleString() || '0'}</div>
                <div className="summary-label">Avg Daily Profit</div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="summary-card">
              <Card.Body className="text-center">
                <div className="summary-value">৳{summary.lastClosing?.toLocaleString() || '0'}</div>
                <div className="summary-label">Last Closing</div>
              </Card.Body>
            </Card>
          </Col>
        </Row>
      )}

      {/* Account Form */}
      {selectedAgent ? (
        <Row>
          <Col lg={6}>
            <Card className="account-card">
              <div className="account-header">
                <h5 className="mb-0">
                  <i className="fas fa-user-tie me-2"></i>
                  Agent Account - {formData.date}
                </h5>
              </div>
              <Card.Body>
                <Form onSubmit={handleSubmit}>
                  <Row>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>Date</Form.Label>
                        <Form.Control
                          type="date"
                          name="date"
                          value={formData.date}
                          onChange={handleInputChange}
                          required
                        />
                      </Form.Group>
                    </Col>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>Day Opening Balance (৳)</Form.Label>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="openingBalance"
                          value={formData.openingBalance}
                          onChange={handleInputChange}
                          placeholder="Enter opening balance"
                          required
                        />
                      </Form.Group>
                    </Col>
                  </Row>

                  <Row>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>Day Closing Balance (৳)</Form.Label>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="closingBalance"
                          value={formData.closingBalance}
                          onChange={handleInputChange}
                          placeholder="Enter closing balance"
                        />
                      </Form.Group>
                    </Col>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>Calculated Profit</Form.Label>
                        <div className="profit-display">
                          ৳{calculateProfit().toLocaleString()}
                        </div>
                      </Form.Group>
                    </Col>
                  </Row>

                  {/* B2B Transaction Fields */}
                  <Row>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>B2B Send (৳)</Form.Label>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="b2bSend"
                          value={formData.b2bSend}
                          onChange={handleInputChange}
                          placeholder="Enter B2B send amount"
                        />
                      </Form.Group>
                    </Col>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>B2B Received (৳)</Form.Label>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="b2bReceived"
                          value={formData.b2bReceived}
                          onChange={handleInputChange}
                          placeholder="Enter B2B received amount"
                        />
                      </Form.Group>
                    </Col>
                  </Row>

                  <Row>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>Amount Received (৳)</Form.Label>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="amountReceived"
                          value={formData.amountReceived}
                          onChange={handleInputChange}
                          placeholder="Enter amount received"
                        />
                      </Form.Group>
                    </Col>
                    <Col md={6}>
                      <Form.Group className="mb-3">
                        <Form.Label>Commission Rate (%)</Form.Label>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="commissionRate"
                          value={formData.commissionRate}
                          onChange={handleInputChange}
                          placeholder="Enter commission rate"
                        />
                      </Form.Group>
                    </Col>
                  </Row>

                <Row>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label className="d-flex justify-content-between align-items-center">
                        SR Received (৳)
                        <div>
                          <Button 
                            variant="outline-success" 
                            size="sm" 
                            onClick={openSRReceivedModal}
                            className="me-1"
                          >
                            <FaPlus /> Add
                          </Button>
                          <Button 
                            variant="outline-info" 
                            size="sm" 
                            onClick={() => setShowSRListModal(true)}
                          >
                            <FaList />
                          </Button>
                        </div>
                      </Form.Label>
                      <InputGroup>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="srReceived"
                          value={formData.srReceived}
                          onChange={handleInputChange}
                          placeholder="Total SR received amount"
                          readOnly
                        />
                        <InputGroup.Text>৳</InputGroup.Text>
                      </InputGroup>
                      {srTransactions.filter(t => t.type === 'received').length > 0 && (
                        <small className="text-muted">
                          {srTransactions.filter(t => t.type === 'received').length} transaction(s) added
                        </small>
                      )}
                    </Form.Group>
                  </Col>
                  <Col md={6}>
                    <Form.Group className="mb-3">
                      <Form.Label className="d-flex justify-content-between align-items-center">
                        SR Given (৳)
                        <div>
                          <Button 
                            variant="outline-danger" 
                            size="sm" 
                            onClick={openSRGivenModal}
                            className="me-1"
                          >
                            <FaPlus /> Add
                          </Button>
                          <Button 
                            variant="outline-info" 
                            size="sm" 
                            onClick={() => setShowSRListModal(true)}
                          >
                            <FaList />
                          </Button>
                        </div>
                      </Form.Label>
                      <InputGroup>
                        <Form.Control
                          type="number"
                          step="0.01"
                          name="srGiven"
                          value={formData.srGiven}
                          onChange={handleInputChange}
                          placeholder="Total SR given amount"
                          readOnly
                        />
                        <InputGroup.Text>৳</InputGroup.Text>
                      </InputGroup>
                      {srTransactions.filter(t => t.type === 'given').length > 0 && (
                        <small className="text-muted">
                          {srTransactions.filter(t => t.type === 'given').length} transaction(s) added
                        </small>
                      )}
                    </Form.Group>
                  </Col>
                </Row>

                <div className="d-flex gap-2">
                  <Button 
                    type="submit" 
                    className="btn-bkash"
                    disabled={loading}
                  >
                    {loading ? <Spinner animation="border" size="sm" /> : 'Save Account'}
                  </Button>
                  
                  {accountData && formData.closingBalance && (
                    <Button 
                      variant="outline-success"
                      onClick={() => setShowCloseModal(true)}
                      disabled={loading}
                    >
                      Close Day
                    </Button>
                  )}
                </div>
              </Form>
            </Card.Body>
          </Card>
        </Col>

        <Col lg={6}>
          <Card className="account-card">
            <Card.Header>
              <h6 className="mb-0">
                <i className="fas fa-chart-line me-2"></i>
                Account Status
              </h6>
            </Card.Header>
            <Card.Body>
              {accountData ? (
                <div>
                  <div className="mb-3">
                    <strong>Day Opening Balance:</strong> ৳{accountData.openingBalance?.toLocaleString() || '0'}
                  </div>
                  <div className="mb-3">
                    <strong>Day Closing Balance:</strong> ৳{(parseFloat(formData.closingBalance) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>B2B Send:</strong> ৳{(parseFloat(formData.b2bSend) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>B2B Received:</strong> ৳{(parseFloat(formData.b2bReceived) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>Amount Received:</strong> ৳{(parseFloat(formData.amountReceived) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>SR Received:</strong> ৳{(parseFloat(formData.srReceived) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>SR Given:</strong> ৳{(parseFloat(formData.srGiven) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>Commission Rate:</strong> {(parseFloat(formData.commissionRate) || 0).toFixed(2)}%
                  </div>
                  <div className="mb-3">
                    <strong>Total Transaction:</strong> ৳{(parseFloat(accountData.totalTransaction) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>Total Commission:</strong> ৳{(parseFloat(accountData.totalCommission) || 0).toLocaleString()}
                  </div>
                  <div className="mb-3">
                    <strong>Profit/Loss:</strong> 
                    <span className={calculateProfit() >= 0 ? 'text-success' : 'text-danger'}>
                      ৳{calculateProfit().toLocaleString()}
                    </span>
                  </div>
                  <div className="mb-3">
                    <strong>Status:</strong> 
                    <Badge bg={accountData.isClosed ? 'success' : 'warning'} className="ms-2">
                      {accountData.isClosed ? 'Closed' : 'Open'}
                    </Badge>
                  </div>
                </div>
              ) : (
                <div className="empty-state">
                  <i className="fas fa-plus-circle"></i>
                  <p>No account data for today. Create a new entry above.</p>
                </div>
              )}
            </Card.Body>
          </Card>
        </Col>
      </Row>
      ) : (
        <Row>
          <Col>
            <Card className="text-center py-5">
              <Card.Body>
                <i className="fas fa-mobile-alt fa-3x text-muted mb-3"></i>
                <h5 className="text-muted">No Agent Selected</h5>
                <p className="text-muted mb-4">Please select a bKash agent number to manage account transactions.</p>
                <Button 
                  variant="primary" 
                  onClick={() => setShowAgentModal(true)}
                >
                  Select Agent Number
                </Button>
              </Card.Body>
            </Card>
          </Col>
        </Row>
      )}

      {/* Agent Selection Modal */}
      <Modal show={showAgentModal} onHide={() => setShowAgentModal(false)} centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <i className="fas fa-mobile-alt me-2"></i>
            Select bKash Agent Number
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          {agentNumbers.length === 0 ? (
            <div className="text-center py-4">
              <i className="fas fa-exclamation-triangle fa-3x text-warning mb-3"></i>
              <h6>No Agent Numbers Available</h6>
              <p className="text-muted">Please add agent numbers in the bKash account management section first.</p>
            </div>
          ) : (
            <div>
              <p className="mb-3">Select an agent number to manage:</p>
              <div className="list-group">
                {agentNumbers.map((agent) => (
                  <button
                    key={agent.id}
                    className={`list-group-item list-group-item-action d-flex justify-content-between align-items-center ${
                      selectedAgent?.id === agent.id ? 'active' : ''
                    }`}
                    onClick={() => {
                      setSelectedAgent(agent);
                      setFormData(prev => ({ ...prev, bkashAccountNumberId: agent.id }));
                      setShowAgentModal(false);
                    }}
                  >
                    <div>
                      <strong>{agent.accountNumber}</strong>
                      <br />
                      <small className="text-muted">{agent.accountName}</small>
                    </div>
                    <div>
                      {agent.isPrimary && (
                        <Badge bg="primary" className="me-2">Primary</Badge>
                      )}
                      <Badge bg={agent.isActive ? 'success' : 'secondary'}>
                        {agent.isActive ? 'Active' : 'Inactive'}
                      </Badge>
                    </div>
                  </button>
                ))}
              </div>
            </div>
          )}
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowAgentModal(false)}>
            Cancel
          </Button>
        </Modal.Footer>
      </Modal>

      {/* Close Day Modal */}
      <Modal show={showCloseModal} onHide={() => setShowCloseModal(false)} centered>
        <Modal.Header closeButton>
          <Modal.Title>Close Day</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <p>Are you sure you want to close the day with the following details?</p>
          <ul>
            <li><strong>Date:</strong> {formData.date}</li>
            <li><strong>Opening Balance:</strong> ৳{(parseFloat(formData.openingBalance) || 0).toLocaleString()}</li>
            <li><strong>Closing Balance:</strong> ৳{(parseFloat(formData.closingBalance) || 0).toLocaleString()}</li>
            <li><strong>B2B Send:</strong> ৳{(parseFloat(formData.b2bSend) || 0).toLocaleString()}</li>
            <li><strong>B2B Received:</strong> ৳{(parseFloat(formData.b2bReceived) || 0).toLocaleString()}</li>
            <li><strong>Profit:</strong> ৳{calculateProfit().toLocaleString()}</li>
          </ul>
          
          {/* B2B Calculation Display */}
          {(parseFloat(formData.b2bSend) > 0 || parseFloat(formData.b2bReceived) > 0) && (
            <div className="alert alert-info">
              <h6><i className="fas fa-calculator me-2"></i>Next Day Opening Balance Calculation:</h6>
              <div className="d-flex justify-content-between">
                <span>Current Closing Balance:</span>
                <span>৳{(parseFloat(formData.closingBalance) || 0).toLocaleString()}</span>
              </div>
              <div className="d-flex justify-content-between text-success">
                <span>+ B2B Received:</span>
                <span>৳{(parseFloat(formData.b2bReceived) || 0).toLocaleString()}</span>
              </div>
              <div className="d-flex justify-content-between text-danger">
                <span>- B2B Send:</span>
                <span>৳{(parseFloat(formData.b2bSend) || 0).toLocaleString()}</span>
              </div>
              <hr className="my-2" />
              <div className="d-flex justify-content-between fw-bold">
                <span>Next Day Opening:</span>
                <span>৳{((parseFloat(formData.closingBalance) || 0) + (parseFloat(formData.b2bReceived) || 0) - (parseFloat(formData.b2bSend) || 0)).toLocaleString()}</span>
              </div>
            </div>
          )}
          
          <p className="text-warning">
            <i className="fas fa-exclamation-triangle me-2"></i>
            This action cannot be undone.
          </p>
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowCloseModal(false)}>
            Cancel
          </Button>
          <Button variant="success" onClick={handleCloseDay} disabled={loading}>
            {loading ? <Spinner animation="border" size="sm" /> : 'Close Day'}
          </Button>
        </Modal.Footer>
      </Modal>

      {/* SR Received Modal */}
      <Modal show={showSRReceivedModal} onHide={closeSRModals} centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <i className="fas fa-arrow-down text-success me-2"></i>
            Add SR Received Transaction
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form>
            <Form.Group className="mb-3">
              <Form.Label>Amount (৳) *</Form.Label>
              <InputGroup>
                <Form.Control
                  type="number"
                  step="0.01"
                  name="amount"
                  value={srTransaction.amount}
                  onChange={handleSRTransactionChange}
                  placeholder="Enter amount received"
                  required
                />
                <InputGroup.Text>৳</InputGroup.Text>
              </InputGroup>
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Description</Form.Label>
              <Form.Control
                type="text"
                name="description"
                value={srTransaction.description}
                onChange={handleSRTransactionChange}
                placeholder="e.g., Cash received from SR office"
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Reference</Form.Label>
              <Form.Control
                type="text"
                name="reference"
                value={srTransaction.reference}
                onChange={handleSRTransactionChange}
                placeholder="Reference number or ID"
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Notes</Form.Label>
              <Form.Control
                as="textarea"
                rows={2}
                name="notes"
                value={srTransaction.notes}
                onChange={handleSRTransactionChange}
                placeholder="Additional notes"
              />
            </Form.Group>
          </Form>
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={closeSRModals}>
            Cancel
          </Button>
          <Button variant="success" onClick={addSRTransaction}>
            <FaPlus className="me-1" /> Add Transaction
          </Button>
        </Modal.Footer>
      </Modal>

      {/* SR Given Modal */}
      <Modal show={showSRGivenModal} onHide={closeSRModals} centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <i className="fas fa-arrow-up text-danger me-2"></i>
            Add SR Given Transaction
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Form>
            <Form.Group className="mb-3">
              <Form.Label>Amount (৳) *</Form.Label>
              <InputGroup>
                <Form.Control
                  type="number"
                  step="0.01"
                  name="amount"
                  value={srTransaction.amount}
                  onChange={handleSRTransactionChange}
                  placeholder="Enter amount given"
                  required
                />
                <InputGroup.Text>৳</InputGroup.Text>
              </InputGroup>
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Description</Form.Label>
              <Form.Control
                type="text"
                name="description"
                value={srTransaction.description}
                onChange={handleSRTransactionChange}
                placeholder="e.g., Cash given to SR office"
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Reference</Form.Label>
              <Form.Control
                type="text"
                name="reference"
                value={srTransaction.reference}
                onChange={handleSRTransactionChange}
                placeholder="Reference number or ID"
              />
            </Form.Group>
            
            <Form.Group className="mb-3">
              <Form.Label>Notes</Form.Label>
              <Form.Control
                as="textarea"
                rows={2}
                name="notes"
                value={srTransaction.notes}
                onChange={handleSRTransactionChange}
                placeholder="Additional notes"
              />
            </Form.Group>
          </Form>
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={closeSRModals}>
            Cancel
          </Button>
          <Button variant="danger" onClick={addSRTransaction}>
            <FaPlus className="me-1" /> Add Transaction
          </Button>
        </Modal.Footer>
      </Modal>

      {/* SR Transactions List Modal */}
      <Modal show={showSRListModal} onHide={() => setShowSRListModal(false)} size="lg" centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <i className="fas fa-list me-2"></i>
            SR Transactions List
          </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          {srTransactions.length === 0 ? (
            <div className="text-center py-4">
              <i className="fas fa-inbox fa-3x text-muted mb-3"></i>
              <p className="text-muted">No SR transactions added yet.</p>
            </div>
          ) : (
            <div className="table-responsive">
              <Table striped bordered hover>
                <thead>
                  <tr>
                    <th>Type</th>
                    <th>Amount</th>
                    <th>Description</th>
                    <th>Reference</th>
                    <th>Time</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {srTransactions.map((transaction) => (
                    <tr key={transaction.id}>
                      <td>
                        <Badge bg={transaction.type === 'received' ? 'success' : 'danger'}>
                          {transaction.type === 'received' ? 'Received' : 'Given'}
                        </Badge>
                      </td>
                      <td className={transaction.type === 'received' ? 'text-success' : 'text-danger'}>
                        ৳{transaction.amount.toLocaleString()}
                      </td>
                      <td>{transaction.description}</td>
                      <td>{transaction.reference || '-'}</td>
                      <td>
                        <small>{transaction.timestamp}</small>
                      </td>
                      <td>
                        <Button
                          variant="outline-danger"
                          size="sm"
                          onClick={() => removeSRTransaction(transaction.id)}
                        >
                          <i className="fas fa-trash"></i>
                        </Button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </Table>
              
              <div className="mt-3">
                <Row>
                  <Col md={6}>
                    <div className="summary-card bg-success text-white p-3 rounded">
                      <h6 className="mb-1">Total Received</h6>
                      <h4 className="mb-0">
                        ৳{srTransactions
                          .filter(t => t.type === 'received')
                          .reduce((sum, t) => sum + t.amount, 0)
                          .toLocaleString()}
                      </h4>
                    </div>
                  </Col>
                  <Col md={6}>
                    <div className="summary-card bg-danger text-white p-3 rounded">
                      <h6 className="mb-1">Total Given</h6>
                      <h4 className="mb-0">
                        ৳{srTransactions
                          .filter(t => t.type === 'given')
                          .reduce((sum, t) => sum + t.amount, 0)
                          .toLocaleString()}
                      </h4>
                    </div>
                  </Col>
                </Row>
              </div>
            </div>
          )}
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowSRListModal(false)}>
            Close
          </Button>
        </Modal.Footer>
      </Modal>
    </div>
  );
};

export default BkashAgentAccount;