import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Table, Button, Modal, Form, Alert, Badge, Dropdown, ButtonGroup } from 'react-bootstrap';
import { FaPlus, FaEdit, FaTrash, FaStar, FaEllipsisV } from 'react-icons/fa';
import axios from 'axios';

const BkashAgentManagement = () => {
  const [agents, setAgents] = useState([]);
  const [showModal, setShowModal] = useState(false);
  const [editingAgent, setEditingAgent] = useState(null);
  const [loading, setLoading] = useState(false);
  const [alert, setAlert] = useState({ show: false, message: '', variant: 'success' });
  const [formData, setFormData] = useState({
    accountNumber: '',
    accountName: '',
    isActive: true,
    isPrimary: false
  });

  useEffect(() => {
    fetchAgents();
  }, []);

  const fetchAgents = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('/api/bkash-account-numbers', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAgents(response.data);
    } catch (error) {
      showAlert('Error fetching agents: ' + error.message, 'danger');
    } finally {
      setLoading(false);
    }
  };

  const showAlert = (message, variant = 'success') => {
    setAlert({ show: true, message, variant });
    setTimeout(() => setAlert({ show: false, message: '', variant: 'success' }), 3000);
  };

  const handleInputChange = (e) => {
    const { name, value, type, checked } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? checked : value
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const agentData = {
        ...formData,
        accountType: 'agent'
      };

      if (editingAgent) {
        await axios.put(`/api/bkash-account-numbers/${editingAgent.id}`, agentData, {
          headers: { Authorization: `Bearer ${token}` }
        });
        showAlert('Agent updated successfully!');
      } else {
        await axios.post('/api/bkash-account-numbers', agentData, {
          headers: { Authorization: `Bearer ${token}` }
        });
        showAlert('Agent created successfully!');
      }

      setShowModal(false);
      setEditingAgent(null);
      setFormData({
        accountNumber: '',
        accountName: '',
        isActive: true,
        isPrimary: false
      });
      fetchAgents();
    } catch (error) {
      showAlert('Error saving agent: ' + error.message, 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleEdit = (agent) => {
    setEditingAgent(agent);
    setFormData({
      accountNumber: agent.accountNumber,
      accountName: agent.accountName,
      isActive: agent.isActive,
      isPrimary: agent.isPrimary
    });
    setShowModal(true);
  };

  const handleDelete = async (agentId) => {
    if (window.confirm('Are you sure you want to delete this agent?')) {
      try {
        setLoading(true);
        const token = localStorage.getItem('token');
        await axios.delete(`/api/bkash-account-numbers/${agentId}`, {
          headers: { Authorization: `Bearer ${token}` }
        });
        showAlert('Agent deleted successfully!');
        fetchAgents();
      } catch (error) {
        showAlert('Error deleting agent: ' + error.message, 'danger');
      } finally {
        setLoading(false);
      }
    }
  };

  const handleSetPrimary = async (agentId) => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      await axios.put(`/api/bkash-account-numbers/${agentId}`, {
        isPrimary: true
      }, {
        headers: { Authorization: `Bearer ${token}` }
      });
      showAlert('Primary agent updated successfully!');
      fetchAgents();
    } catch (error) {
      showAlert('Error setting primary agent: ' + error.message, 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleCloseModal = () => {
    setShowModal(false);
    setEditingAgent(null);
    setFormData({
      accountNumber: '',
      accountName: '',
      isActive: true,
      isPrimary: false
    });
  };

  return (
    <Container fluid className="px-2 px-md-3">
      <Row>
        <Col>
          <Card className="shadow-sm">
            <Card.Header className="d-flex flex-column flex-sm-row justify-content-between align-items-start align-items-sm-center gap-2">
              <h4 className="mb-0">bKash Agent Management</h4>
              <Button 
                variant="primary" 
                onClick={() => setShowModal(true)}
                disabled={loading}
                className="w-100 w-sm-auto"
              >
                <FaPlus className="me-2" />
                <span className="d-none d-sm-inline">Add New Agent</span>
                <span className="d-sm-none">Add Agent</span>
              </Button>
            </Card.Header>
            <Card.Body>
              {alert.show && (
                <Alert variant={alert.variant} className="mb-3">
                  {alert.message}
                </Alert>
              )}

              {loading && !showModal ? (
                <div className="text-center">
                  <div className="spinner-border" role="status">
                    <span className="visually-hidden">Loading...</span>
                  </div>
                </div>
              ) : (
                <>
                  {/* Desktop Table View */}
                  <div className="d-none d-lg-block">
                    <Table striped bordered hover responsive>
                      <thead>
                        <tr>
                          <th>Account Number</th>
                          <th>Account Name</th>
                          <th>Status</th>
                          <th>Primary</th>
                          <th>Actions</th>
                        </tr>
                      </thead>
                      <tbody>
                        {agents.length === 0 ? (
                          <tr>
                            <td colSpan="5" className="text-center py-4">
                              <div className="text-muted">
                                <FaPlus className="mb-2" style={{fontSize: '2rem'}} />
                                <p>No agents found. Add your first agent to get started.</p>
                              </div>
                            </td>
                          </tr>
                        ) : (
                          agents.map(agent => (
                            <tr key={agent.id}>
                              <td className="fw-bold">{agent.accountNumber}</td>
                              <td>{agent.accountName}</td>
                              <td>
                                <Badge bg={agent.isActive ? 'success' : 'secondary'}>
                                  {agent.isActive ? 'Active' : 'Inactive'}
                                </Badge>
                              </td>
                              <td>
                                {agent.isPrimary ? (
                                  <Badge bg="warning">
                                    <FaStar className="me-1" />
                                    Primary
                                  </Badge>
                                ) : (
                                  <Button
                                    size="sm"
                                    variant="outline-warning"
                                    onClick={() => handleSetPrimary(agent.id)}
                                    disabled={loading}
                                  >
                                    Set Primary
                                  </Button>
                                )}
                              </td>
                              <td>
                                <ButtonGroup size="sm">
                                  <Button
                                    variant="outline-primary"
                                    onClick={() => handleEdit(agent)}
                                    disabled={loading}
                                  >
                                    <FaEdit />
                                  </Button>
                                  <Button
                                    variant="outline-danger"
                                    onClick={() => handleDelete(agent.id)}
                                    disabled={loading || agent.isPrimary}
                                  >
                                    <FaTrash />
                                  </Button>
                                </ButtonGroup>
                              </td>
                            </tr>
                          ))
                        )}
                      </tbody>
                    </Table>
                  </div>

                  {/* Mobile Card View */}
                  <div className="d-lg-none">
                    {agents.length === 0 ? (
                      <div className="text-center py-5">
                        <FaPlus className="mb-3 text-muted" style={{fontSize: '3rem'}} />
                        <h5 className="text-muted">No agents found</h5>
                        <p className="text-muted">Add your first agent to get started.</p>
                      </div>
                    ) : (
                      <Row className="g-3">
                        {agents.map(agent => (
                          <Col key={agent.id} xs={12} sm={6}>
                            <Card className="h-100 shadow-sm">
                              <Card.Body>
                                <div className="d-flex justify-content-between align-items-start mb-2">
                                  <div className="flex-grow-1">
                                    <h6 className="fw-bold mb-1">{agent.accountNumber}</h6>
                                    <p className="text-muted mb-2 small">{agent.accountName}</p>
                                  </div>
                                  <Dropdown>
                                    <Dropdown.Toggle 
                                      variant="outline-secondary" 
                                      size="sm" 
                                      className="border-0"
                                      disabled={loading}
                                    >
                                      <FaEllipsisV />
                                    </Dropdown.Toggle>
                                    <Dropdown.Menu>
                                      <Dropdown.Item onClick={() => handleEdit(agent)}>
                                        <FaEdit className="me-2" /> Edit
                                      </Dropdown.Item>
                                      {!agent.isPrimary && (
                                        <>
                                          <Dropdown.Item 
                                            onClick={() => handleSetPrimary(agent.id)}
                                            disabled={loading}
                                          >
                                            <FaStar className="me-2" /> Set Primary
                                          </Dropdown.Item>
                                          <Dropdown.Divider />
                                          <Dropdown.Item 
                                            className="text-danger"
                                            onClick={() => handleDelete(agent.id)}
                                            disabled={loading}
                                          >
                                            <FaTrash className="me-2" /> Delete
                                          </Dropdown.Item>
                                        </>
                                      )}
                                    </Dropdown.Menu>
                                  </Dropdown>
                                </div>
                                
                                <div className="d-flex justify-content-between align-items-center">
                                  <Badge bg={agent.isActive ? 'success' : 'secondary'}>
                                    {agent.isActive ? 'Active' : 'Inactive'}
                                  </Badge>
                                  {agent.isPrimary && (
                                    <Badge bg="warning">
                                      <FaStar className="me-1" />
                                      Primary
                                    </Badge>
                                  )}
                                </div>
                              </Card.Body>
                            </Card>
                          </Col>
                        ))}
                      </Row>
                    )}
                  </div>
                </>
              )}
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Add/Edit Modal */}
      <Modal show={showModal} onHide={handleCloseModal} centered>
        <Modal.Header closeButton>
          <Modal.Title>
            <FaPlus className="me-2" />
            {editingAgent ? 'Edit Agent' : 'Add New Agent'}
          </Modal.Title>
        </Modal.Header>
        <Form onSubmit={handleSubmit}>
          <Modal.Body className="px-4">
            <Row>
              <Col xs={12}>
                <Form.Group className="mb-4">
                  <Form.Label className="fw-semibold">Account Number</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountNumber"
                    value={formData.accountNumber}
                    onChange={handleInputChange}
                    placeholder="e.g., 01712345678"
                    className="form-control-lg"
                    required
                  />
                  <Form.Text className="text-muted">
                    Enter the bKash account number (11 digits)
                  </Form.Text>
                </Form.Group>
              </Col>
            </Row>

            <Row>
              <Col xs={12}>
                <Form.Group className="mb-4">
                  <Form.Label className="fw-semibold">Account Name</Form.Label>
                  <Form.Control
                    type="text"
                    name="accountName"
                    value={formData.accountName}
                    onChange={handleInputChange}
                    placeholder="e.g., Main Agent Account"
                    className="form-control-lg"
                    required
                  />
                  <Form.Text className="text-muted">
                    Give this agent account a descriptive name
                  </Form.Text>
                </Form.Group>
              </Col>
            </Row>

            <Row>
              <Col xs={12} sm={6}>
                <Form.Group className="mb-3">
                  <div className="d-flex align-items-center">
                    <Form.Check
                      type="switch"
                      id="isActive-switch"
                      name="isActive"
                      label="Active Status"
                      checked={formData.isActive}
                      onChange={handleInputChange}
                      className="form-check-lg"
                    />
                  </div>
                  <Form.Text className="text-muted small">
                    Active agents can be used for transactions
                  </Form.Text>
                </Form.Group>
              </Col>
              <Col xs={12} sm={6}>
                <Form.Group className="mb-3">
                  <div className="d-flex align-items-center">
                    <Form.Check
                      type="switch"
                      id="isPrimary-switch"
                      name="isPrimary"
                      label="Primary Agent"
                      checked={formData.isPrimary}
                      onChange={handleInputChange}
                      className="form-check-lg"
                    />
                  </div>
                  <Form.Text className="text-muted small">
                    Only one primary agent allowed
                  </Form.Text>
                </Form.Group>
              </Col>
            </Row>

            {formData.isPrimary && (
              <Alert variant="warning" className="mt-3">
                <small>
                  <strong>Note:</strong> Setting this as primary will remove primary status from other agents.
                </small>
              </Alert>
            )}
          </Modal.Body>
          <Modal.Footer className="d-flex flex-column flex-sm-row gap-2">
            <Button 
              variant="outline-secondary" 
              onClick={handleCloseModal}
              className="w-100 w-sm-auto order-2 order-sm-1"
            >
              Cancel
            </Button>
            <Button 
              variant="primary" 
              type="submit" 
              disabled={loading}
              className="w-100 w-sm-auto order-1 order-sm-2"
            >
              {loading ? (
                <>
                  <div className="spinner-border spinner-border-sm me-2" role="status">
                    <span className="visually-hidden">Loading...</span>
                  </div>
                  Saving...
                </>
              ) : (
                <>
                  <FaPlus className="me-2" />
                  {editingAgent ? 'Update Agent' : 'Add Agent'}
                </>
              )}
            </Button>
          </Modal.Footer>
        </Form>
      </Modal>
    </Container>
  );
};

export default BkashAgentManagement;