import React, { useState, useEffect } from 'react';
import { Row, Col, Card, Form, Button, Table, Badge, Spinner, Modal, Tab, Tabs } from 'react-bootstrap';
import axios from 'axios';
import { useAuth } from '../../contexts/AuthContext';

const BkashMerchantAccount = ({ showAlert }) => {
  const { currentUser } = useAuth();
  const [accountData, setAccountData] = useState(null);
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(false);
  const [activeTab, setActiveTab] = useState('account');
  const [formData, setFormData] = useState({
    openingBalance: '',
    closingBalance: '',
    date: new Date().toISOString().split('T')[0]
  });
  const [transactionForm, setTransactionForm] = useState({
    type: 'merchant_sale',
    amount: '',
    description: '',
    date: new Date().toISOString().split('T')[0]
  });
  const [showCloseModal, setShowCloseModal] = useState(false);
  const [summary, setSummary] = useState(null);

  useEffect(() => {
    // Only fetch data if user is authenticated
    if (currentUser) {
      fetchTodayAccount();
      fetchTodayTransactions();
      fetchSummary();
    }
  }, [currentUser]);

  const fetchTodayAccount = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-merchant/today', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setAccountData(response.data);
      if (response.data) {
        setFormData({
          openingBalance: response.data.openingBalance || '',
          closingBalance: response.data.closingBalance || '',
          date: response.data.date || new Date().toISOString().split('T')[0]
        });
      }
    } catch (error) {
      console.error('Error fetching today account:', error);
      showAlert('Error fetching account data', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const fetchTodayTransactions = async () => {
    try {
      const token = localStorage.getItem('token');
      const today = new Date().toISOString().split('T')[0];
      const response = await axios.get(`http://localhost:5000/api/bkash-merchant/${today}/transactions`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setTransactions(response.data);
    } catch (error) {
      console.error('Error fetching transactions:', error);
    }
  };

  const fetchSummary = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/bkash-merchant/', {
        headers: { Authorization: `Bearer ${token}` }
      });
      setSummary(response.data);
    } catch (error) {
      console.error('Error fetching summary:', error);
    }
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleTransactionChange = (e) => {
    const { name, value } = e.target;
    setTransactionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleAccountSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...formData,
        openingBalance: parseFloat(formData.openingBalance) || 0,
        closingBalance: parseFloat(formData.closingBalance) || 0
      };

      await axios.put(`http://localhost:5000/api/bkash-merchant/${formData.date}`, submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Merchant account updated successfully!', 'success');
      fetchTodayAccount();
      fetchSummary();
    } catch (error) {
      console.error('Error updating account:', error);
      showAlert('Error updating account', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleTransactionSubmit = async (e) => {
    e.preventDefault();
    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      const submitData = {
        ...transactionForm,
        amount: parseFloat(transactionForm.amount) || 0
      };

      await axios.post(`http://localhost:5000/api/bkash-merchant/${transactionForm.date}/transactions`, submitData, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Transaction added successfully!', 'success');
      setTransactionForm({
        type: 'merchant_sale',
        amount: '',
        description: '',
        date: new Date().toISOString().split('T')[0]
      });
      fetchTodayTransactions();
      fetchTodayAccount();
    } catch (error) {
      console.error('Error adding transaction:', error);
      showAlert('Error adding transaction', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteTransaction = async (transactionId) => {
    if (!window.confirm('Are you sure you want to delete this transaction?')) {
      return;
    }

    try {
      setLoading(true);
      const token = localStorage.getItem('token');
      
      await axios.delete(`http://localhost:5000/api/bkash-merchant/transactions/${transactionId}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      showAlert('Transaction deleted successfully!', 'success');
      fetchTodayTransactions();
      fetchTodayAccount();
    } catch (error) {
      console.error('Error deleting transaction:', error);
      showAlert('Error deleting transaction', 'danger');
    } finally {
      setLoading(false);
    }
  };

  const calculateBalance = () => {
    if (!accountData) return 0;
    const opening = parseFloat(accountData.openingBalance) || 0;
    const closing = parseFloat(formData.closingBalance) || parseFloat(accountData.closingBalance) || 0;
    return closing - opening;
  };

  const getTransactionTypeLabel = (type) => {
    switch (type) {
      case 'merchant_sale': return 'Sale';
      case 'merchant_refund': return 'Refund';
      case 'merchant_withdrawal': return 'Withdrawal';
      case 'merchant_fee': return 'Fee';
      default: return type;
    }
  };

  const getTransactionTypeBadge = (type) => {
    switch (type) {
      case 'merchant_sale': return 'success';
      case 'merchant_refund': return 'warning';
      case 'merchant_withdrawal': return 'info';
      case 'merchant_fee': return 'danger';
      default: return 'secondary';
    }
  };

  return (
    <div className="bkash-merchant-account">
      {/* Header */}
      <div className="page-header">
        <div className="d-flex justify-content-between align-items-center">
          <div>
            <h4 className="page-title">
              <i className="fas fa-store me-2"></i>
              Bkash Merchant Account
            </h4>
            <p className="page-subtitle">Manage your merchant account transactions and balance</p>
          </div>
          <div className="header-actions">
            <Badge bg="info" className="me-2">
              Today: {new Date().toLocaleDateString()}
            </Badge>
            {accountData && (
              <Badge bg={accountData.isCompleted ? 'success' : 'warning'}>
                {accountData.isCompleted ? 'Completed' : 'In Progress'}
              </Badge>
            )}
          </div>
        </div>
      </div>

      {/* Main Content Tabs */}
      <Tabs activeKey={activeTab} onSelect={(k) => setActiveTab(k)} className="mb-3">
        <Tab eventKey="account" title="Account Management">
          <Row>
            <Col lg={6}>
              <Card className="account-card">
                <div className="account-header">
                  <h5 className="mb-0">
                    <i className="fas fa-store me-2"></i>
                    Merchant Account - {formData.date}
                  </h5>
                </div>
                <Card.Body>
                  <Form onSubmit={handleAccountSubmit}>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Date</Form.Label>
                          <Form.Control
                            type="date"
                            name="date"
                            value={formData.date}
                            onChange={handleInputChange}
                            required
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Opening Balance (৳)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="openingBalance"
                            value={formData.openingBalance}
                            onChange={handleInputChange}
                            placeholder="Enter opening balance"
                            required
                          />
                        </Form.Group>
                      </Col>
                    </Row>

                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Closing Balance (৳)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="closingBalance"
                            value={formData.closingBalance}
                            onChange={handleInputChange}
                            placeholder="Enter closing balance"
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Net Change</Form.Label>
                          <div className="profit-display">
                            ৳{calculateBalance().toLocaleString()}
                          </div>
                        </Form.Group>
                      </Col>
                    </Row>

                    <div className="d-flex gap-2">
                      <Button 
                        type="submit" 
                        className="btn-bkash"
                        disabled={loading}
                      >
                        {loading ? <Spinner animation="border" size="sm" /> : 'Save Account'}
                      </Button>
                    </div>
                  </Form>
                </Card.Body>
              </Card>
            </Col>

            <Col lg={6}>
              <Card className="account-card">
                <Card.Header>
                  <h6 className="mb-0">
                    <i className="fas fa-chart-line me-2"></i>
                    Account Status
                  </h6>
                </Card.Header>
                <Card.Body>
                  {accountData ? (
                    <div>
                      <div className="mb-3">
                        <strong>Opening Balance:</strong> ৳{accountData.openingBalance?.toLocaleString() || '0'}
                      </div>
                      <div className="mb-3">
                        <strong>Total Sales:</strong> ৳{accountData.totalSales?.toLocaleString() || '0'}
                      </div>
                      <div className="mb-3">
                        <strong>Total Refunds:</strong> ৳{accountData.totalRefunds?.toLocaleString() || '0'}
                      </div>
                      <div className="mb-3">
                        <strong>Merchant Fee:</strong> ৳{accountData.merchantFee?.toLocaleString() || '0'}
                      </div>
                      <div className="mb-3">
                        <strong>Net Revenue:</strong> 
                        <span className="text-success">
                          ৳{accountData.netRevenue?.toLocaleString() || '0'}
                        </span>
                      </div>
                      <div className="mb-3">
                        <strong>Current Closing:</strong> ৳{(parseFloat(formData.closingBalance) || 0).toLocaleString()}
                      </div>
                      <div className="mb-3">
                        <strong>Net Change:</strong> 
                        <span className={calculateBalance() >= 0 ? 'text-success' : 'text-danger'}>
                          ৳{calculateBalance().toLocaleString()}
                        </span>
                      </div>
                    </div>
                  ) : (
                    <div className="empty-state">
                      <i className="fas fa-plus-circle"></i>
                      <p>No account data for today. Create a new entry above.</p>
                    </div>
                  )}
                </Card.Body>
              </Card>
            </Col>
          </Row>
        </Tab>

        <Tab eventKey="transactions" title="Transactions">
          <Row>
            <Col lg={6}>
              <Card className="transaction-form">
                <Card.Header>
                  <h6 className="mb-0">
                    <i className="fas fa-plus me-2"></i>
                    Add Transaction
                  </h6>
                </Card.Header>
                <Card.Body>
                  <Form onSubmit={handleTransactionSubmit}>
                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Transaction Type</Form.Label>
                          <Form.Select
                            name="type"
                            value={transactionForm.type}
                            onChange={handleTransactionChange}
                            required
                          >
                            <option value="merchant_sale">Sale</option>
                            <option value="merchant_refund">Refund</option>
                            <option value="merchant_withdrawal">Withdrawal</option>
                            <option value="merchant_fee">Fee</option>
                          </Form.Select>
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Amount (৳)</Form.Label>
                          <Form.Control
                            type="number"
                            step="0.01"
                            name="amount"
                            value={transactionForm.amount}
                            onChange={handleTransactionChange}
                            placeholder="Enter amount"
                            required
                          />
                        </Form.Group>
                      </Col>
                    </Row>

                    <Row>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Date</Form.Label>
                          <Form.Control
                            type="date"
                            name="date"
                            value={transactionForm.date}
                            onChange={handleTransactionChange}
                            required
                          />
                        </Form.Group>
                      </Col>
                      <Col md={6}>
                        <Form.Group className="mb-3">
                          <Form.Label>Description</Form.Label>
                          <Form.Control
                            type="text"
                            name="description"
                            value={transactionForm.description}
                            onChange={handleTransactionChange}
                            placeholder="Enter description"
                          />
                        </Form.Group>
                      </Col>
                    </Row>

                    <Button 
                      type="submit" 
                      className="btn-bkash"
                      disabled={loading}
                    >
                      {loading ? <Spinner animation="border" size="sm" /> : 'Add Transaction'}
                    </Button>
                  </Form>
                </Card.Body>
              </Card>
            </Col>

            <Col lg={6}>
              <Card className="transaction-list">
                <Card.Header>
                  <h6 className="mb-0">
                    <i className="fas fa-list me-2"></i>
                    Today's Transactions ({transactions.length})
                  </h6>
                </Card.Header>
                <Card.Body>
                  {transactions.length > 0 ? (
                    <div className="table-responsive">
                      <Table striped hover size="sm">
                        <thead>
                          <tr>
                            <th>Type</th>
                            <th>Amount</th>
                            <th>Description</th>
                            <th>Actions</th>
                          </tr>
                        </thead>
                        <tbody>
                          {transactions.map((transaction) => (
                            <tr key={transaction.id}>
                              <td>
                                <Badge bg={getTransactionTypeBadge(transaction.type)}>
                                  {getTransactionTypeLabel(transaction.type)}
                                </Badge>
                              </td>
                              <td>৳{transaction.amount?.toLocaleString()}</td>
                              <td>{transaction.description || '-'}</td>
                              <td>
                                <Button
                                  variant="outline-danger"
                                  size="sm"
                                  onClick={() => handleDeleteTransaction(transaction.id)}
                                  disabled={loading}
                                >
                                  <i className="fas fa-trash"></i>
                                </Button>
                              </td>
                            </tr>
                          ))}
                        </tbody>
                      </Table>
                    </div>
                  ) : (
                    <div className="empty-state">
                      <i className="fas fa-receipt"></i>
                      <p>No transactions for today</p>
                    </div>
                  )}
                </Card.Body>
              </Card>
            </Col>
          </Row>
        </Tab>
      </Tabs>
    </div>
  );
};

export default BkashMerchantAccount;