import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Tab, Tabs, Table, Button, Form, Alert, Spinner } from 'react-bootstrap';
import { FaMobileAlt, FaDownload, FaCalendarAlt, FaChartLine } from 'react-icons/fa';
import axios from 'axios';
import { useAuth } from '../../contexts/AuthContext';
import './BkashReports.css';

const BkashReports = () => {
  const { currentUser } = useAuth();
  const [activeTab, setActiveTab] = useState('daily');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [reports, setReports] = useState({
    daily: null,
    weekly: null,
    monthly: null,
    comparison: null
  });
  const [dateRange, setDateRange] = useState({
    startDate: new Date().toISOString().split('T')[0],
    endDate: new Date().toISOString().split('T')[0]
  });

  useEffect(() => {
    // Only fetch data if user is authenticated
    if (currentUser) {
      fetchReport(activeTab);
    }
  }, [activeTab, currentUser]);

  const fetchReport = async (reportType) => {
    setLoading(true);
    setError('');
    
    try {
      const token = localStorage.getItem('token');
      const headers = { Authorization: `Bearer ${token}` };
      
      let url = '';
      switch (reportType) {
        case 'daily':
          url = `http://localhost:5000/api/bkash-reports/daily?date=${dateRange.startDate}`;
          break;
        case 'weekly':
          url = `http://localhost:5000/api/bkash-reports/weekly?date=${dateRange.startDate}`;
          break;
        case 'monthly':
          url = `http://localhost:5000/api/bkash-reports/monthly?date=${dateRange.startDate}`;
          break;
        case 'comparison':
          url = `http://localhost:5000/api/bkash-reports/comparison?startDate=${dateRange.startDate}&endDate=${dateRange.endDate}`;
          break;
        default:
          return;
      }

      const response = await axios.get(url, { headers });
      setReports(prev => ({
        ...prev,
        [reportType]: response.data
      }));
    } catch (error) {
      console.error('Error fetching report:', error);
      setError('Failed to fetch report data. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const handleDateChange = (field, value) => {
    setDateRange(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const downloadReport = async (reportType) => {
    try {
      const token = localStorage.getItem('token');
      const headers = { Authorization: `Bearer ${token}` };
      
      let url = '';
      switch (reportType) {
        case 'daily':
          url = `http://localhost:5000/api/bkash-reports/daily/export?date=${dateRange.startDate}`;
          break;
        case 'weekly':
          url = `http://localhost:5000/api/bkash-reports/weekly/export?date=${dateRange.startDate}`;
          break;
        case 'monthly':
          url = `http://localhost:5000/api/bkash-reports/monthly/export?date=${dateRange.startDate}`;
          break;
        case 'comparison':
          url = `http://localhost:5000/api/bkash-reports/comparison/export?startDate=${dateRange.startDate}&endDate=${dateRange.endDate}`;
          break;
        default:
          return;
      }

      const response = await axios.get(url, { 
        headers,
        responseType: 'blob'
      });
      
      const blob = new Blob([response.data], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
      const downloadUrl = window.URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = downloadUrl;
      link.download = `bkash-${reportType}-report-${new Date().toISOString().split('T')[0]}.xlsx`;
      document.body.appendChild(link);
      link.click();
      link.remove();
      window.URL.revokeObjectURL(downloadUrl);
    } catch (error) {
      console.error('Error downloading report:', error);
      setError('Failed to download report. Please try again.');
    }
  };

  const formatCurrency = (amount) => {
    return `৳${parseFloat(amount || 0).toLocaleString()}`;
  };

  const renderDailyReport = () => {
    const data = reports.daily;
    if (!data) return null;

    return (
      <div>
        <Row className="mb-4">
          <Col md={3}>
            <Card className="text-center border-primary">
              <Card.Body>
                <h5 className="text-primary">Agent Opening</h5>
                <h3>{formatCurrency(data.agentOpening)}</h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="text-center border-success">
              <Card.Body>
                <h5 className="text-success">Agent Closing</h5>
                <h3>{formatCurrency(data.agentClosing)}</h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="text-center border-warning">
              <Card.Body>
                <h5 className="text-warning">Daily Profit</h5>
                <h3 className={data.profit >= 0 ? 'text-success' : 'text-danger'}>
                  {formatCurrency(data.profit)}
                </h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="text-center border-info">
              <Card.Body>
                <h5 className="text-info">Transactions</h5>
                <h3>{data.totalTransactions}</h3>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {data.transactions && data.transactions.length > 0 && (
          <Card>
            <Card.Header>
              <h5>Transaction Details</h5>
            </Card.Header>
            <Card.Body>
              <Table responsive striped hover>
                <thead>
                  <tr>
                    <th>Time</th>
                    <th>Type</th>
                    <th>Amount</th>
                    <th>Description</th>
                    <th>Account</th>
                  </tr>
                </thead>
                <tbody>
                  {data.transactions.map((transaction, index) => (
                    <tr key={index}>
                      <td>{new Date(transaction.createdAt).toLocaleTimeString()}</td>
                      <td>
                        <span className={`badge ${transaction.type === 'cash_in' ? 'bg-success' : 'bg-danger'}`}>
                          {transaction.type.replace('_', ' ').toUpperCase()}
                        </span>
                      </td>
                      <td>{formatCurrency(transaction.amount)}</td>
                      <td>{transaction.description}</td>
                      <td>{transaction.accountType}</td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </Card.Body>
          </Card>
        )}
      </div>
    );
  };

  const renderWeeklyReport = () => {
    const data = reports.weekly;
    if (!data) return null;

    return (
      <div>
        <Row className="mb-4">
          <Col md={4}>
            <Card className="text-center border-primary">
              <Card.Body>
                <h5 className="text-primary">Total Profit</h5>
                <h3 className={data.totalProfit >= 0 ? 'text-success' : 'text-danger'}>
                  {formatCurrency(data.totalProfit)}
                </h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={4}>
            <Card className="text-center border-success">
              <Card.Body>
                <h5 className="text-success">Total Transactions</h5>
                <h3>{data.totalTransactions}</h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={4}>
            <Card className="text-center border-info">
              <Card.Body>
                <h5 className="text-info">Average Daily Profit</h5>
                <h3 className={data.averageDailyProfit >= 0 ? 'text-success' : 'text-danger'}>
                  {formatCurrency(data.averageDailyProfit)}
                </h3>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {data.dailyBreakdown && data.dailyBreakdown.length > 0 && (
          <Card>
            <Card.Header>
              <h5>Daily Breakdown</h5>
            </Card.Header>
            <Card.Body>
              <Table responsive striped hover>
                <thead>
                  <tr>
                    <th>Date</th>
                    <th>Opening</th>
                    <th>Closing</th>
                    <th>Profit</th>
                    <th>Transactions</th>
                  </tr>
                </thead>
                <tbody>
                  {data.dailyBreakdown.map((day, index) => (
                    <tr key={index}>
                      <td>{new Date(day.date).toLocaleDateString()}</td>
                      <td>{formatCurrency(day.opening)}</td>
                      <td>{formatCurrency(day.closing)}</td>
                      <td className={day.profit >= 0 ? 'text-success' : 'text-danger'}>
                        {formatCurrency(day.profit)}
                      </td>
                      <td>{day.transactions}</td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </Card.Body>
          </Card>
        )}
      </div>
    );
  };

  const renderMonthlyReport = () => {
    const data = reports.monthly;
    if (!data) return null;

    return (
      <div>
        <Row className="mb-4">
          <Col md={3}>
            <Card className="text-center border-primary">
              <Card.Body>
                <h5 className="text-primary">Total Profit</h5>
                <h3 className={data.totalProfit >= 0 ? 'text-success' : 'text-danger'}>
                  {formatCurrency(data.totalProfit)}
                </h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="text-center border-success">
              <Card.Body>
                <h5 className="text-success">Total Transactions</h5>
                <h3>{data.totalTransactions}</h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="text-center border-warning">
              <Card.Body>
                <h5 className="text-warning">Best Day Profit</h5>
                <h3 className="text-success">{formatCurrency(data.bestDayProfit)}</h3>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="text-center border-info">
              <Card.Body>
                <h5 className="text-info">Average Daily</h5>
                <h3 className={data.averageDailyProfit >= 0 ? 'text-success' : 'text-danger'}>
                  {formatCurrency(data.averageDailyProfit)}
                </h3>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {data.weeklyBreakdown && data.weeklyBreakdown.length > 0 && (
          <Card>
            <Card.Header>
              <h5>Weekly Breakdown</h5>
            </Card.Header>
            <Card.Body>
              <Table responsive striped hover>
                <thead>
                  <tr>
                    <th>Week</th>
                    <th>Start Date</th>
                    <th>End Date</th>
                    <th>Profit</th>
                    <th>Transactions</th>
                  </tr>
                </thead>
                <tbody>
                  {data.weeklyBreakdown.map((week, index) => (
                    <tr key={index}>
                      <td>Week {index + 1}</td>
                      <td>{new Date(week.startDate).toLocaleDateString()}</td>
                      <td>{new Date(week.endDate).toLocaleDateString()}</td>
                      <td className={week.profit >= 0 ? 'text-success' : 'text-danger'}>
                        {formatCurrency(week.profit)}
                      </td>
                      <td>{week.transactions}</td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </Card.Body>
          </Card>
        )}
      </div>
    );
  };

  return (
    <Container fluid className="bkash-reports-container">
      <Row className="mb-4">
        <Col>
          <div className="d-flex justify-content-between align-items-center">
            <h2>
              <FaMobileAlt className="me-2 text-primary" />
              Bkash Reports
            </h2>
            <Button 
              variant="outline-primary" 
              onClick={() => downloadReport(activeTab)}
              disabled={loading}
            >
              <FaDownload className="me-2" />
              Download Report
            </Button>
          </div>
        </Col>
      </Row>

      <Row className="mb-4">
        <Col md={6}>
          <Form.Group>
            <Form.Label>
              <FaCalendarAlt className="me-2" />
              {activeTab === 'comparison' ? 'Start Date' : 'Date'}
            </Form.Label>
            <Form.Control
              type="date"
              value={dateRange.startDate}
              onChange={(e) => handleDateChange('startDate', e.target.value)}
            />
          </Form.Group>
        </Col>
        {activeTab === 'comparison' && (
          <Col md={6}>
            <Form.Group>
              <Form.Label>End Date</Form.Label>
              <Form.Control
                type="date"
                value={dateRange.endDate}
                onChange={(e) => handleDateChange('endDate', e.target.value)}
              />
            </Form.Group>
          </Col>
        )}
      </Row>

      <Row className="mb-3">
        <Col>
          <Button 
            variant="primary" 
            onClick={() => fetchReport(activeTab)}
            disabled={loading}
          >
            <FaChartLine className="me-2" />
            Generate Report
          </Button>
        </Col>
      </Row>

      {error && (
        <Alert variant="danger" dismissible onClose={() => setError('')}>
          {error}
        </Alert>
      )}

      <Tabs
        activeKey={activeTab}
        onSelect={(k) => setActiveTab(k)}
        className="mb-4"
      >
        <Tab eventKey="daily" title="Daily Report">
          {loading ? (
            <div className="text-center py-5">
              <Spinner animation="border" variant="primary" />
              <p className="mt-2">Loading daily report...</p>
            </div>
          ) : (
            renderDailyReport()
          )}
        </Tab>

        <Tab eventKey="weekly" title="Weekly Report">
          {loading ? (
            <div className="text-center py-5">
              <Spinner animation="border" variant="primary" />
              <p className="mt-2">Loading weekly report...</p>
            </div>
          ) : (
            renderWeeklyReport()
          )}
        </Tab>

        <Tab eventKey="monthly" title="Monthly Report">
          {loading ? (
            <div className="text-center py-5">
              <Spinner animation="border" variant="primary" />
              <p className="mt-2">Loading monthly report...</p>
            </div>
          ) : (
            renderMonthlyReport()
          )}
        </Tab>

        <Tab eventKey="comparison" title="Comparison Report">
          {loading ? (
            <div className="text-center py-5">
              <Spinner animation="border" variant="primary" />
              <p className="mt-2">Loading comparison report...</p>
            </div>
          ) : (
            <div className="text-center py-5">
              <p>Comparison report functionality coming soon...</p>
            </div>
          )}
        </Tab>
      </Tabs>
    </Container>
  );
};

export default BkashReports;