import React, { useState, useEffect } from 'react';
import {
  Container,
  Row,
  Col,
  Card,
  Form,
  Button,
  Table,
  Modal,
  Alert,
  Spinner,
  Badge,
  Tab,
  Tabs,
  Nav,
  InputGroup
} from 'react-bootstrap';
import { useAuth } from '../../contexts/AuthContext';

const DSODeliveries = ({ showAlert }) => {
  const { currentUser } = useAuth();
  
  const [deliveryForm, setDeliveryForm] = useState({
    date: new Date().toISOString().split('T')[0],
    dsoName: '',
    receivedBy: currentUser?.username || 'Current User',
    amountPaid: '',
    cardsReceivedTotal: '',
    cardTypes: [{ type: '', quantity: '' }],
    simsReceived: '',
    kitsReceived: '',
    notes: ''
  });

  const [allocationForm, setAllocationForm] = useState({
    deliveryId: '',
    cardsAllocated: '',
    cardTypesAllocated: [{ type: '', quantity: '' }],
    simsAllocated: '',
    kitsAllocated: '',
    notes: ''
  });

  const [deliveries, setDeliveries] = useState([]);
  const [dsoNames, setDSONames] = useState([]);
  const [loading, setLoading] = useState(false);
  const [showConfirmModal, setShowConfirmModal] = useState(false);
  const [confirmData, setConfirmData] = useState(null);
  const [errors, setErrors] = useState({});
  const [activeTab, setActiveTab] = useState('delivery');

  useEffect(() => {
    fetchDeliveries();
    fetchDSONames();
  }, []);

  // Update receivedBy when currentUser becomes available
  useEffect(() => {
    if (currentUser?.username) {
      setDeliveryForm(prev => ({
        ...prev,
        receivedBy: currentUser.username
      }));
    }
  }, [currentUser]);

  const fetchDeliveries = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('http://localhost:5000/api/dso/deliveries', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      const data = await response.json();
      if (response.ok) {
        setDeliveries(data.deliveries || []);
      }
    } catch (error) {
      console.error('Error fetching deliveries:', error);
    }
  };

  const fetchDSONames = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('http://localhost:5000/api/dso/dso-names', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      const data = await response.json();
      if (response.ok) {
        setDSONames(data || []);
      }
    } catch (error) {
      console.error('Error fetching DSO names:', error);
    }
  };

  const handleDeliveryInputChange = (e) => {
    const { name, value } = e.target;
    setDeliveryForm(prev => ({
      ...prev,
      [name]: value
    }));
    
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({
        ...prev,
        [name]: ''
      }));
    }
  };

  const handleCardTypeChange = (index, field, value) => {
    const newCardTypes = [...deliveryForm.cardTypes];
    newCardTypes[index] = {
      ...newCardTypes[index],
      [field]: value
    };
    setDeliveryForm(prev => ({
      ...prev,
      cardTypes: newCardTypes
    }));
  };

  const addCardTypeRow = () => {
    setDeliveryForm(prev => ({
      ...prev,
      cardTypes: [...prev.cardTypes, { type: '', quantity: '' }]
    }));
  };

  const removeCardTypeRow = (index) => {
    if (deliveryForm.cardTypes.length > 1) {
      const newCardTypes = deliveryForm.cardTypes.filter((_, i) => i !== index);
      setDeliveryForm(prev => ({
        ...prev,
        cardTypes: newCardTypes
      }));
    }
  };

  const validateDeliveryForm = () => {
    const newErrors = {};

    if (!deliveryForm.dsoName.trim()) {
      newErrors.dsoName = 'DSO name is required';
    }

    if (!deliveryForm.receivedBy.trim()) {
      newErrors.receivedBy = 'Received by is required';
    }

    const amountPaid = parseFloat(deliveryForm.amountPaid);
    if (isNaN(amountPaid) || amountPaid < 0) {
      newErrors.amountPaid = 'Amount must be a non-negative number';
    }

    const cardsTotal = parseInt(deliveryForm.cardsReceivedTotal) || 0;
    const simsTotal = parseInt(deliveryForm.simsReceived) || 0;
    const kitsTotal = parseInt(deliveryForm.kitsReceived) || 0;

    if (cardsTotal === 0 && simsTotal === 0 && kitsTotal === 0) {
      newErrors.general = 'At least one of Cards, SIMs, or Kits must be received (quantity > 0)';
    }

    // Validate card types if cards are received
    if (cardsTotal > 0) {
      // Filter out completely empty card type rows (both type and quantity empty)
      const filledCardTypes = deliveryForm.cardTypes.filter(ct => 
        ct.type.trim() !== '' || (ct.quantity && parseInt(ct.quantity) > 0)
      );

      // If user has filled any card type information, validate it
      if (filledCardTypes.length > 0) {
        const cardTypesTotal = filledCardTypes.reduce((sum, cardType) => {
          return sum + (parseInt(cardType.quantity) || 0);
        }, 0);

        if (cardTypesTotal !== cardsTotal) {
          newErrors.cardTypes = 'Sum of card type quantities must equal total cards received';
        }

        // Check for incomplete card types (type filled but no quantity, or quantity filled but no type)
        const hasIncompleteCardTypes = filledCardTypes.some(ct => 
          (ct.type.trim() !== '' && (!ct.quantity || parseInt(ct.quantity) <= 0)) ||
          (ct.type.trim() === '' && ct.quantity && parseInt(ct.quantity) > 0)
        );
        if (hasIncompleteCardTypes) {
          newErrors.cardTypes = 'All card types must have both a name and positive quantity';
        }
      }
      // If no card types are filled, that's okay - user doesn't want to break down by type
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleDeliverySubmit = (saveAndAllocate = false) => {
    if (!validateDeliveryForm()) {
      return;
    }

    // Filter out completely empty card type rows (both type and quantity empty)
    const filledCardTypes = deliveryForm.cardTypes.filter(ct => 
      ct.type.trim() !== '' || (ct.quantity && parseInt(ct.quantity) > 0)
    );

    const deliveryData = {
      ...deliveryForm,
      amountPaid: parseFloat(deliveryForm.amountPaid) || 0,
      cardsReceivedTotal: parseInt(deliveryForm.cardsReceivedTotal) || 0,
      simsReceived: parseInt(deliveryForm.simsReceived) || 0,
      kitsReceived: parseInt(deliveryForm.kitsReceived) || 0,
      cardTypes: filledCardTypes,
      receivedBy: deliveryForm.receivedBy,
      createdBy: currentUser?.username || 'Current User'
    };

    setConfirmData({ deliveryData, saveAndAllocate });
    setShowConfirmModal(true);
  };

  const confirmDelivery = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('http://localhost:5000/api/dso/deliveries', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(confirmData.deliveryData)
      });

      const data = await response.json();

      if (response.ok) {
        showAlert('DSO delivery saved successfully!', 'success');
        
        // Reset form
        setDeliveryForm({
          date: new Date().toISOString().split('T')[0],
          dsoName: '',
          receivedBy: currentUser?.username || 'Current User',
          amountPaid: '',
          cardsReceivedTotal: '',
          cardTypes: [{ type: '', quantity: '' }],
          simsReceived: '',
          kitsReceived: '',
          notes: ''
        });

        fetchDeliveries();
        
        if (confirmData.saveAndAllocate) {
          setAllocationForm(prev => ({
            ...prev,
            deliveryId: data.delivery.id
          }));
          setActiveTab('allocation');
        }
      } else {
        showAlert(data.message || 'Error saving delivery', 'danger');
      }
    } catch (error) {
      console.error('Error saving delivery:', error);
      showAlert('Error saving delivery', 'danger');
    } finally {
      setLoading(false);
      setShowConfirmModal(false);
    }
  };

  const handleAllocationInputChange = (e) => {
    const { name, value } = e.target;
    setAllocationForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleAllocationCardTypeChange = (index, field, value) => {
    const newCardTypes = [...allocationForm.cardTypesAllocated];
    newCardTypes[index] = {
      ...newCardTypes[index],
      [field]: value
    };
    setAllocationForm(prev => ({
      ...prev,
      cardTypesAllocated: newCardTypes
    }));
  };

  const addAllocationCardTypeRow = () => {
    setAllocationForm(prev => ({
      ...prev,
      cardTypesAllocated: [...prev.cardTypesAllocated, { type: '', quantity: '' }]
    }));
  };

  const removeAllocationCardTypeRow = (index) => {
    if (allocationForm.cardTypesAllocated.length > 1) {
      const newCardTypes = allocationForm.cardTypesAllocated.filter((_, i) => i !== index);
      setAllocationForm(prev => ({
        ...prev,
        cardTypesAllocated: newCardTypes
      }));
    }
  };

  const handleAllocationSubmit = async () => {
    setLoading(true);
    try {
      const allocationData = {
        ...allocationForm,
        cardsAllocated: parseInt(allocationForm.cardsAllocated) || 0,
        simsAllocated: parseInt(allocationForm.simsAllocated) || 0,
        kitsAllocated: parseInt(allocationForm.kitsAllocated) || 0,
        cardTypesAllocated: allocationForm.cardTypesAllocated.filter(ct => ct.type.trim() !== ''),
        allocatedBy: 'Current User' // This should be set from auth context
      };

      const token = localStorage.getItem('token');
      const response = await fetch('http://localhost:5000/api/dso/allocations', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(allocationData)
      });

      const data = await response.json();

      if (response.ok) {
        showAlert('DSO allocation saved successfully!', 'success');
        
        // Reset allocation form
        setAllocationForm({
          deliveryId: '',
          cardsAllocated: '',
          cardTypesAllocated: [{ type: '', quantity: '' }],
          simsAllocated: '',
          kitsAllocated: '',
          notes: ''
        });

        fetchDeliveries();
      } else {
        showAlert(data.message || 'Error saving allocation', 'danger');
      }
    } catch (error) {
      console.error('Error saving allocation:', error);
      showAlert('Error saving allocation', 'danger');
    } finally {
      setLoading(false);
    }
  };

  return (
    <Container fluid>
      <Row className="mb-4">
        <Col>
          <h3 className="text-primary">
            <i className="fas fa-truck me-2"></i>
            DSO Deliveries & Allocations
          </h3>
          <p className="text-muted">Manage deliveries from Distribution Sales Officers and allocate stock for sales</p>
        </Col>
      </Row>

      <Tabs activeKey={activeTab} onSelect={setActiveTab} className="mb-4">
        <Tab eventKey="delivery" title={<><i className="fas fa-plus-circle me-2"></i>New Delivery</>}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">Create DSO Delivery</h5>
            </Card.Header>
            <Card.Body>
              {errors.general && (
                <Alert variant="danger">{errors.general}</Alert>
              )}
              
              <Row>
                <Col md={4}>
                  <Form.Group className="mb-3">
                    <Form.Label>Date</Form.Label>
                    <Form.Control
                      type="date"
                      name="date"
                      value={deliveryForm.date}
                      onChange={handleDeliveryInputChange}
                    />
                  </Form.Group>
                </Col>
                <Col md={4}>
                  <Form.Group className="mb-3">
                    <Form.Label>DSO Name *</Form.Label>
                    <Form.Control
                      type="text"
                      name="dsoName"
                      value={deliveryForm.dsoName}
                      onChange={handleDeliveryInputChange}
                      list="dso-names"
                      isInvalid={!!errors.dsoName}
                    />
                    <datalist id="dso-names">
                      {dsoNames.map((name, index) => (
                        <option key={index} value={name} />
                      ))}
                    </datalist>
                    <Form.Control.Feedback type="invalid">
                      {errors.dsoName}
                    </Form.Control.Feedback>
                  </Form.Group>
                </Col>
                <Col md={4}>
                  <Form.Group className="mb-3">
                    <Form.Label>Received By *</Form.Label>
                    <Form.Control
                      type="text"
                      name="receivedBy"
                      value={deliveryForm.receivedBy}
                      onChange={handleDeliveryInputChange}
                      isInvalid={!!errors.receivedBy}
                    />
                    <Form.Control.Feedback type="invalid">
                      {errors.receivedBy}
                    </Form.Control.Feedback>
                  </Form.Group>
                </Col>
              </Row>

              <Row>
                <Col md={3}>
                  <Form.Group className="mb-3">
                    <Form.Label>DSO Payment (৳) *</Form.Label>
                    <InputGroup>
                      <InputGroup.Text>৳</InputGroup.Text>
                      <Form.Control
                        type="number"
                        step="0.01"
                        min="0"
                        name="amountPaid"
                        value={deliveryForm.amountPaid}
                        onChange={handleDeliveryInputChange}
                        isInvalid={!!errors.amountPaid}
                      />
                      <Form.Control.Feedback type="invalid">
                        {errors.amountPaid}
                      </Form.Control.Feedback>
                    </InputGroup>
                  </Form.Group>
                </Col>
                <Col md={3}>
                  <Form.Group className="mb-3">
                    <Form.Label>Cards Received</Form.Label>
                    <Form.Control
                      type="number"
                      min="0"
                      name="cardsReceivedTotal"
                      value={deliveryForm.cardsReceivedTotal}
                      onChange={handleDeliveryInputChange}
                    />
                  </Form.Group>
                </Col>
                <Col md={3}>
                  <Form.Group className="mb-3">
                    <Form.Label>SIMs Received</Form.Label>
                    <Form.Control
                      type="number"
                      min="0"
                      name="simsReceived"
                      value={deliveryForm.simsReceived}
                      onChange={handleDeliveryInputChange}
                    />
                  </Form.Group>
                </Col>
                <Col md={3}>
                  <Form.Group className="mb-3">
                    <Form.Label>Kits Received</Form.Label>
                    <Form.Control
                      type="number"
                      min="0"
                      name="kitsReceived"
                      value={deliveryForm.kitsReceived}
                      onChange={handleDeliveryInputChange}
                    />
                  </Form.Group>
                </Col>
              </Row>

              {parseInt(deliveryForm.cardsReceivedTotal) > 0 && (
                <Card className="mb-3">
                  <Card.Header>
                    <h6 className="mb-0">Card Types Received</h6>
                  </Card.Header>
                  <Card.Body>
                    {errors.cardTypes && (
                      <Alert variant="danger" className="mb-3">{errors.cardTypes}</Alert>
                    )}
                    {deliveryForm.cardTypes.map((cardType, index) => (
                      <Row key={index} className="mb-2">
                        <Col md={6}>
                          <Form.Control
                            type="text"
                            placeholder="Card Type"
                            value={cardType.type}
                            onChange={(e) => handleCardTypeChange(index, 'type', e.target.value)}
                          />
                        </Col>
                        <Col md={4}>
                          <Form.Control
                            type="number"
                            min="1"
                            placeholder="Quantity"
                            value={cardType.quantity}
                            onChange={(e) => handleCardTypeChange(index, 'quantity', e.target.value)}
                          />
                        </Col>
                        <Col md={2}>
                          <Button
                            variant="outline-danger"
                            size="sm"
                            onClick={() => removeCardTypeRow(index)}
                            disabled={deliveryForm.cardTypes.length === 1}
                          >
                            <i className="fas fa-trash"></i>
                          </Button>
                        </Col>
                      </Row>
                    ))}
                    <Button
                      variant="outline-primary"
                      size="sm"
                      onClick={addCardTypeRow}
                    >
                      <i className="fas fa-plus me-2"></i>Add Card Type Row
                    </Button>
                  </Card.Body>
                </Card>
              )}

              <Form.Group className="mb-3">
                <Form.Label>Notes</Form.Label>
                <Form.Control
                  as="textarea"
                  rows={3}
                  name="notes"
                  value={deliveryForm.notes}
                  onChange={handleDeliveryInputChange}
                  placeholder="Additional notes about the delivery..."
                />
              </Form.Group>

              <div className="d-flex gap-2">
                <Button
                  variant="primary"
                  onClick={() => handleDeliverySubmit(false)}
                  disabled={loading}
                >
                  <i className="fas fa-save me-2"></i>
                  Save as Delivery
                </Button>
                <Button
                  variant="success"
                  onClick={() => handleDeliverySubmit(true)}
                  disabled={loading}
                >
                  <i className="fas fa-share me-2"></i>
                  Save & Allocate to Sales
                </Button>
              </div>
            </Card.Body>
          </Card>
        </Tab>

        <Tab eventKey="allocation" title={<><i className="fas fa-share-alt me-2"></i>Allocation</>}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">Allocate for Sales</h5>
            </Card.Header>
            <Card.Body>
              <Row>
                <Col md={6}>
                  <Form.Group className="mb-3">
                    <Form.Label>Select Delivery (Optional)</Form.Label>
                    <Form.Select
                      name="deliveryId"
                      value={allocationForm.deliveryId}
                      onChange={handleAllocationInputChange}
                    >
                      <option value="">Select a delivery...</option>
                      {deliveries.map(delivery => (
                        <option key={delivery.id} value={delivery.id}>
                          {delivery.date} - {delivery.dsoName} (Cards: {delivery.cardsReceivedTotal}, SIMs: {delivery.simsReceived}, Kits: {delivery.kitsReceived})
                        </option>
                      ))}
                    </Form.Select>
                  </Form.Group>
                </Col>
              </Row>

              <Row>
                <Col md={4}>
                  <Form.Group className="mb-3">
                    <Form.Label>Cards Allocated for Sale Today</Form.Label>
                    <Form.Control
                      type="number"
                      min="0"
                      name="cardsAllocated"
                      value={allocationForm.cardsAllocated}
                      onChange={handleAllocationInputChange}
                    />
                  </Form.Group>
                </Col>
                <Col md={4}>
                  <Form.Group className="mb-3">
                    <Form.Label>SIMs Allocated</Form.Label>
                    <div className="d-flex align-items-center">
                      <Form.Control
                        type="number"
                        min="0"
                        name="simsAllocated"
                        value={allocationForm.simsAllocated}
                        onChange={handleAllocationInputChange}
                      />
                      <Form.Check
                        type="checkbox"
                        label="Mark for Sale Today"
                        className="ms-2"
                        checked={parseInt(allocationForm.simsAllocated) > 0}
                        onChange={(e) => {
                          if (e.target.checked && !allocationForm.simsAllocated) {
                            setAllocationForm(prev => ({ ...prev, simsAllocated: '1' }));
                          }
                        }}
                      />
                    </div>
                  </Form.Group>
                </Col>
                <Col md={4}>
                  <Form.Group className="mb-3">
                    <Form.Label>Kits Allocated</Form.Label>
                    <div className="d-flex align-items-center">
                      <Form.Control
                        type="number"
                        min="0"
                        name="kitsAllocated"
                        value={allocationForm.kitsAllocated}
                        onChange={handleAllocationInputChange}
                      />
                      <Form.Check
                        type="checkbox"
                        label="Mark for Sale Today"
                        className="ms-2"
                        checked={parseInt(allocationForm.kitsAllocated) > 0}
                        onChange={(e) => {
                          if (e.target.checked && !allocationForm.kitsAllocated) {
                            setAllocationForm(prev => ({ ...prev, kitsAllocated: '1' }));
                          }
                        }}
                      />
                    </div>
                  </Form.Group>
                </Col>
              </Row>

              {parseInt(allocationForm.cardsAllocated) > 0 && (
                <Card className="mb-3">
                  <Card.Header>
                    <h6 className="mb-0">Card Types Allocation Breakdown (Optional)</h6>
                  </Card.Header>
                  <Card.Body>
                    {allocationForm.cardTypesAllocated.map((cardType, index) => (
                      <Row key={index} className="mb-2">
                        <Col md={6}>
                          <Form.Control
                            type="text"
                            placeholder="Card Type"
                            value={cardType.type}
                            onChange={(e) => handleAllocationCardTypeChange(index, 'type', e.target.value)}
                          />
                        </Col>
                        <Col md={4}>
                          <Form.Control
                            type="number"
                            min="1"
                            placeholder="Quantity"
                            value={cardType.quantity}
                            onChange={(e) => handleAllocationCardTypeChange(index, 'quantity', e.target.value)}
                          />
                        </Col>
                        <Col md={2}>
                          <Button
                            variant="outline-danger"
                            size="sm"
                            onClick={() => removeAllocationCardTypeRow(index)}
                            disabled={allocationForm.cardTypesAllocated.length === 1}
                          >
                            <i className="fas fa-trash"></i>
                          </Button>
                        </Col>
                      </Row>
                    ))}
                    <Button
                      variant="outline-primary"
                      size="sm"
                      onClick={addAllocationCardTypeRow}
                    >
                      <i className="fas fa-plus me-2"></i>Add Card Type Row
                    </Button>
                  </Card.Body>
                </Card>
              )}

              <Form.Group className="mb-3">
                <Form.Label>Notes</Form.Label>
                <Form.Control
                  as="textarea"
                  rows={2}
                  name="notes"
                  value={allocationForm.notes}
                  onChange={handleAllocationInputChange}
                  placeholder="Additional notes about the allocation..."
                />
              </Form.Group>

              <Button
                variant="success"
                onClick={handleAllocationSubmit}
                disabled={loading}
              >
                <i className="fas fa-check me-2"></i>
                Save Allocation
              </Button>
            </Card.Body>
          </Card>
        </Tab>

        <Tab eventKey="history" title={<><i className="fas fa-history me-2"></i>Recent Deliveries</>}>
          <Card>
            <Card.Header>
              <h5 className="mb-0">Recent DSO Deliveries</h5>
            </Card.Header>
            <Card.Body>
              <Table responsive striped>
                <thead>
                  <tr>
                    <th>Date</th>
                    <th>DSO Name</th>
                    <th>Amount Paid</th>
                    <th>Cards</th>
                    <th>SIMs</th>
                    <th>Kits</th>
                    <th>Received By</th>
                    <th>Status</th>
                  </tr>
                </thead>
                <tbody>
                  {deliveries.slice(0, 10).map(delivery => (
                    <tr key={delivery.id}>
                      <td>{delivery.date}</td>
                      <td>{delivery.dsoName}</td>
                      <td>৳{parseFloat(delivery.amountPaid).toFixed(2)}</td>
                      <td>{delivery.cardsReceivedTotal}</td>
                      <td>{delivery.simsReceived}</td>
                      <td>{delivery.kitsReceived}</td>
                      <td>{delivery.receivedBy}</td>
                      <td>
                        <Badge bg={delivery.status === 'active' ? 'success' : 'secondary'}>
                          {delivery.status}
                        </Badge>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </Card.Body>
          </Card>
        </Tab>
      </Tabs>

      {/* Confirmation Modal */}
      <Modal show={showConfirmModal} onHide={() => setShowConfirmModal(false)}>
        <Modal.Header closeButton>
          <Modal.Title>Confirm DSO Delivery</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          {confirmData && (
            <div>
              <h6>Delivery Summary:</h6>
              <ul>
                <li><strong>DSO:</strong> {confirmData.deliveryData.dsoName}</li>
                <li><strong>Amount Paid:</strong> ৳{parseFloat(confirmData.deliveryData.amountPaid).toFixed(2)}</li>
                <li><strong>Cards:</strong> {confirmData.deliveryData.cardsReceivedTotal}</li>
                <li><strong>SIMs:</strong> {confirmData.deliveryData.simsReceived}</li>
                <li><strong>Kits:</strong> {confirmData.deliveryData.kitsReceived}</li>
              </ul>
              {confirmData.saveAndAllocate && (
                <Alert variant="info">
                  <i className="fas fa-info-circle me-2"></i>
                  After saving, you will be redirected to the allocation tab.
                </Alert>
              )}
            </div>
          )}
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowConfirmModal(false)}>
            Cancel
          </Button>
          <Button variant="primary" onClick={confirmDelivery} disabled={loading}>
            {loading ? 'Saving...' : 'Confirm & Save'}
          </Button>
        </Modal.Footer>
      </Modal>
    </Container>
  );
};

export default DSODeliveries;