import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Button, Table, Modal, Form, Alert, Badge, InputGroup } from 'react-bootstrap';
import { FaPlus, FaEdit, FaTrash, FaSearch, FaFileExport, FaPrint, FaEye } from 'react-icons/fa';
import './CustomerManagement.css';

const CustomerManagement = () => {
  const [customers, setCustomers] = useState([]);
  const [showModal, setShowModal] = useState(false);
  const [editingCustomer, setEditingCustomer] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedArea, setSelectedArea] = useState('all');
  const [showAlert, setShowAlert] = useState({ show: false, message: '', type: '' });
  const [formData, setFormData] = useState({
    customerName: '',
    meterNumber: '',
    phoneNumber: '',
    address: '',
    area: '',
    connectionType: 'residential',
    tariffRate: 5.5,
    status: 'active'
  });

  // Mock data for customers
  useEffect(() => {
    const mockCustomers = [
      {
        id: 1,
        customerName: 'আব্দুল করিম',
        meterNumber: 'PB001234',
        phoneNumber: '01712345678',
        address: 'গ্রাম: রামপুর, ডাকঘর: রামপুর',
        area: 'রামপুর',
        connectionType: 'residential',
        tariffRate: 5.5,
        status: 'active',
        joinDate: '2023-01-15',
        lastBillAmount: 850
      },
      {
        id: 2,
        customerName: 'ফাতেমা খাতুন',
        meterNumber: 'PB001235',
        phoneNumber: '01812345679',
        address: 'গ্রাম: রামপুর, ডাকঘর: রামপুর',
        area: 'রামপুর',
        connectionType: 'residential',
        tariffRate: 5.5,
        status: 'active',
        joinDate: '2023-02-20',
        lastBillAmount: 650
      },
      {
        id: 3,
        customerName: 'মোহাম্মদ আলী',
        meterNumber: 'PB001236',
        phoneNumber: '01912345680',
        address: 'গ্রাম: কালিগঞ্জ, ডাকঘর: কালিগঞ্জ',
        area: 'কালিগঞ্জ',
        connectionType: 'commercial',
        tariffRate: 8.5,
        status: 'active',
        joinDate: '2022-12-10',
        lastBillAmount: 2500
      }
    ];
    setCustomers(mockCustomers);
  }, []);

  // Filter customers based on search and area
  const filteredCustomers = customers.filter(customer => {
    const matchesSearch = customer.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         customer.meterNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         customer.phoneNumber.includes(searchTerm);
    const matchesArea = selectedArea === 'all' || customer.area === selectedArea;
    return matchesSearch && matchesArea;
  });

  // Handle form submission
  const handleSubmit = (e) => {
    e.preventDefault();
    
    if (editingCustomer) {
      // Update existing customer
      setCustomers(customers.map(customer => 
        customer.id === editingCustomer.id 
          ? { ...formData, id: editingCustomer.id, joinDate: editingCustomer.joinDate }
          : customer
      ));
      setShowAlert({ show: true, message: 'গ্রাহকের তথ্য সফলভাবে আপডেট হয়েছে।', type: 'success' });
    } else {
      // Add new customer
      const newCustomer = {
        ...formData,
        id: Date.now(),
        joinDate: new Date().toISOString().split('T')[0],
        lastBillAmount: 0
      };
      setCustomers([...customers, newCustomer]);
      setShowAlert({ show: true, message: 'নতুন গ্রাহক সফলভাবে যোগ করা হয়েছে।', type: 'success' });
    }
    
    handleCloseModal();
    setTimeout(() => setShowAlert({ show: false, message: '', type: '' }), 3000);
  };

  // Handle edit
  const handleEdit = (customer) => {
    setEditingCustomer(customer);
    setFormData(customer);
    setShowModal(true);
  };

  // Handle delete
  const handleDelete = (customerId) => {
    if (window.confirm('আপনি কি নিশ্চিত যে এই গ্রাহককে মুছে ফেলতে চান?')) {
      setCustomers(customers.filter(customer => customer.id !== customerId));
      setShowAlert({ show: true, message: 'গ্রাহক সফলভাবে মুছে ফেলা হয়েছে।', type: 'info' });
      setTimeout(() => setShowAlert({ show: false, message: '', type: '' }), 3000);
    }
  };

  // Handle modal close
  const handleCloseModal = () => {
    setShowModal(false);
    setEditingCustomer(null);
    setFormData({
      customerName: '',
      meterNumber: '',
      phoneNumber: '',
      address: '',
      area: '',
      connectionType: 'residential',
      tariffRate: 5.5,
      status: 'active'
    });
  };

  // Export to CSV
  const exportToCSV = () => {
    const csvContent = [
      ['গ্রাহকের নাম', 'মিটার নম্বর', 'ফোন নম্বর', 'ঠিকানা', 'এলাকা', 'সংযোগের ধরন', 'ট্যারিফ রেট', 'অবস্থা'],
      ...filteredCustomers.map(customer => [
        customer.customerName,
        customer.meterNumber,
        customer.phoneNumber,
        customer.address,
        customer.area,
        customer.connectionType === 'residential' ? 'আবাসিক' : 'বাণিজ্যিক',
        customer.tariffRate,
        customer.status === 'active' ? 'সক্রিয়' : 'নিষ্ক্রিয়'
      ])
    ].map(row => row.join(',')).join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'palli_bidyut_customers.csv';
    link.click();
  };

  return (
    <Container fluid>
      <Row className="mb-4">
        <Col>
          <h2>পল্লী বিদ্যুৎ - গ্রাহক ব্যবস্থাপনা</h2>
        </Col>
      </Row>

      {showAlert.show && (
        <Alert variant={showAlert.type} className="mb-3">
          {showAlert.message}
        </Alert>
      )}

      {/* Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="summary-card total">
            <Card.Body className="text-center">
              <h3>{customers.length}</h3>
              <p>মোট গ্রাহক</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card active">
            <Card.Body className="text-center">
              <h3>{customers.filter(c => c.status === 'active').length}</h3>
              <p>সক্রিয় গ্রাহক</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card residential">
            <Card.Body className="text-center">
              <h3>{customers.filter(c => c.connectionType === 'residential').length}</h3>
              <p>আবাসিক সংযোগ</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card commercial">
            <Card.Body className="text-center">
              <h3>{customers.filter(c => c.connectionType === 'commercial').length}</h3>
              <p>বাণিজ্যিক সংযোগ</p>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Search and Filter */}
      <Row className="mb-3">
        <Col md={4}>
          <InputGroup>
            <Form.Control
              type="text"
              placeholder="গ্রাহক খুঁজুন..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            <Button variant="outline-secondary">
              <FaSearch />
            </Button>
          </InputGroup>
        </Col>
        <Col md={3}>
          <Form.Select value={selectedArea} onChange={(e) => setSelectedArea(e.target.value)}>
            <option value="all">সব এলাকা</option>
            <option value="রামপুর">রামপুর</option>
            <option value="কালিগঞ্জ">কালিগঞ্জ</option>
            <option value="বাজারপাড়া">বাজারপাড়া</option>
          </Form.Select>
        </Col>
        <Col md={5} className="text-end">
          <Button variant="primary" className="me-2" onClick={() => setShowModal(true)}>
            <FaPlus className="me-1" /> নতুন গ্রাহক
          </Button>
          <Button variant="success" className="me-2" onClick={exportToCSV}>
            <FaFileExport className="me-1" /> এক্সপোর্ট
          </Button>
          <Button variant="info" onClick={() => window.print()}>
            <FaPrint className="me-1" /> প্রিন্ট
          </Button>
        </Col>
      </Row>

      {/* Customer Table */}
      <Card>
        <Card.Body>
          <Table responsive striped hover>
            <thead>
              <tr>
                <th>গ্রাহকের নাম</th>
                <th>মিটার নম্বর</th>
                <th>ফোন নম্বর</th>
                <th>এলাকা</th>
                <th>সংযোগের ধরন</th>
                <th>ট্যারিফ রেট</th>
                <th>অবস্থা</th>
                <th>শেষ বিল</th>
                <th>কার্যক্রম</th>
              </tr>
            </thead>
            <tbody>
              {filteredCustomers.map(customer => (
                <tr key={customer.id}>
                  <td>
                    <strong>{customer.customerName}</strong>
                    <br />
                    <small className="text-muted">যোগদান: {new Date(customer.joinDate).toLocaleDateString('bn-BD')}</small>
                  </td>
                  <td>{customer.meterNumber}</td>
                  <td>{customer.phoneNumber}</td>
                  <td>{customer.area}</td>
                  <td>
                    <Badge bg={customer.connectionType === 'residential' ? 'primary' : 'warning'}>
                      {customer.connectionType === 'residential' ? 'আবাসিক' : 'বাণিজ্যিক'}
                    </Badge>
                  </td>
                  <td>৳{customer.tariffRate}/ইউনিট</td>
                  <td>
                    <Badge bg={customer.status === 'active' ? 'success' : 'danger'}>
                      {customer.status === 'active' ? 'সক্রিয়' : 'নিষ্ক্রিয়'}
                    </Badge>
                  </td>
                  <td>৳{customer.lastBillAmount}</td>
                  <td>
                    <div className="d-flex gap-1">
                      <Button variant="outline-info" size="sm" title="বিস্তারিত">
                        <FaEye />
                      </Button>
                      <Button variant="outline-primary" size="sm" onClick={() => handleEdit(customer)} title="সম্পাদনা">
                        <FaEdit />
                      </Button>
                      <Button variant="outline-danger" size="sm" onClick={() => handleDelete(customer.id)} title="মুছে ফেলুন">
                        <FaTrash />
                      </Button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </Table>

          {filteredCustomers.length === 0 && (
            <div className="text-center py-4">
              <p className="text-muted">কোন গ্রাহক পাওয়া যায়নি।</p>
            </div>
          )}
        </Card.Body>
      </Card>

      {/* Add/Edit Customer Modal */}
      <Modal show={showModal} onHide={handleCloseModal} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>
            {editingCustomer ? 'গ্রাহকের তথ্য সম্পাদনা' : 'নতুন গ্রাহক যোগ করুন'}
          </Modal.Title>
        </Modal.Header>
        <Form onSubmit={handleSubmit}>
          <Modal.Body>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>গ্রাহকের নাম *</Form.Label>
                  <Form.Control
                    type="text"
                    value={formData.customerName}
                    onChange={(e) => setFormData({...formData, customerName: e.target.value})}
                    required
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>মিটার নম্বর *</Form.Label>
                  <Form.Control
                    type="text"
                    value={formData.meterNumber}
                    onChange={(e) => setFormData({...formData, meterNumber: e.target.value})}
                    required
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>ফোন নম্বর *</Form.Label>
                  <Form.Control
                    type="tel"
                    value={formData.phoneNumber}
                    onChange={(e) => setFormData({...formData, phoneNumber: e.target.value})}
                    required
                  />
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>এলাকা *</Form.Label>
                  <Form.Select
                    value={formData.area}
                    onChange={(e) => setFormData({...formData, area: e.target.value})}
                    required
                  >
                    <option value="">এলাকা নির্বাচন করুন</option>
                    <option value="রামপুর">রামপুর</option>
                    <option value="কালিগঞ্জ">কালিগঞ্জ</option>
                    <option value="বাজারপাড়া">বাজারপাড়া</option>
                  </Form.Select>
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={12}>
                <Form.Group className="mb-3">
                  <Form.Label>ঠিকানা *</Form.Label>
                  <Form.Control
                    as="textarea"
                    rows={2}
                    value={formData.address}
                    onChange={(e) => setFormData({...formData, address: e.target.value})}
                    required
                  />
                </Form.Group>
              </Col>
            </Row>
            <Row>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>সংযোগের ধরন</Form.Label>
                  <Form.Select
                    value={formData.connectionType}
                    onChange={(e) => setFormData({...formData, connectionType: e.target.value})}
                  >
                    <option value="residential">আবাসিক</option>
                    <option value="commercial">বাণিজ্যিক</option>
                  </Form.Select>
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>ট্যারিফ রেট (৳/ইউনিট)</Form.Label>
                  <Form.Control
                    type="number"
                    step="0.1"
                    value={formData.tariffRate}
                    onChange={(e) => setFormData({...formData, tariffRate: parseFloat(e.target.value)})}
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group className="mb-3">
                  <Form.Label>অবস্থা</Form.Label>
                  <Form.Select
                    value={formData.status}
                    onChange={(e) => setFormData({...formData, status: e.target.value})}
                  >
                    <option value="active">সক্রিয়</option>
                    <option value="inactive">নিষ্ক্রিয়</option>
                  </Form.Select>
                </Form.Group>
              </Col>
            </Row>
          </Modal.Body>
          <Modal.Footer>
            <Button variant="secondary" onClick={handleCloseModal}>
              বাতিল
            </Button>
            <Button variant="primary" type="submit">
              {editingCustomer ? 'আপডেট করুন' : 'সংরক্ষণ করুন'}
            </Button>
          </Modal.Footer>
        </Form>
      </Modal>
    </Container>
  );
};

export default CustomerManagement;