import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Button, Table, Modal, Form, Alert, Badge, InputGroup, ProgressBar } from 'react-bootstrap';
import { FaSearch, FaExclamationTriangle, FaCalendarAlt, FaFileExport, FaPrint, FaPhone, FaSms, FaEnvelope } from 'react-icons/fa';
import './PendingBills.css';

const PendingBills = () => {
  const [pendingBills, setPendingBills] = useState([]);
  const [showReminderModal, setShowReminderModal] = useState(false);
  const [selectedBills, setSelectedBills] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedArea, setSelectedArea] = useState('all');
  const [dueDateFilter, setDueDateFilter] = useState('all');
  const [showAlert, setShowAlert] = useState({ show: false, message: '', type: '' });
  const [reminderData, setReminderData] = useState({
    method: 'sms',
    message: 'আপনার বিদ্যুৎ বিলের পরিশোধের শেষ তারিখ আসন্ন। অনুগ্রহ করে যথাসময়ে পরিশোধ করুন।'
  });

  // Mock data for pending bills
  useEffect(() => {
    const mockPendingBills = [
      {
        id: 1,
        customerName: 'আব্দুল করিম',
        meterNumber: 'PB001234',
        phoneNumber: '01712345678',
        area: 'রামপুর',
        billMonth: '2024-01',
        billAmount: 825,
        dueDate: '2024-02-15',
        daysOverdue: 5,
        connectionType: 'residential',
        lastReminderSent: '2024-02-10',
        reminderCount: 2
      },
      {
        id: 2,
        customerName: 'ফাতেমা খাতুন',
        meterNumber: 'PB001235',
        phoneNumber: '01812345679',
        area: 'রামপুর',
        billMonth: '2024-01',
        billAmount: 660,
        dueDate: '2024-02-15',
        daysOverdue: 5,
        connectionType: 'residential',
        lastReminderSent: null,
        reminderCount: 0
      },
      {
        id: 3,
        customerName: 'মোহাম্মদ আলী',
        meterNumber: 'PB001236',
        phoneNumber: '01912345680',
        area: 'কালিগঞ্জ',
        billMonth: '2024-01',
        billAmount: 2550,
        dueDate: '2024-02-15',
        daysOverdue: 5,
        connectionType: 'commercial',
        lastReminderSent: '2024-02-12',
        reminderCount: 1
      },
      {
        id: 4,
        customerName: 'সালমা আক্তার',
        meterNumber: 'PB001238',
        phoneNumber: '01512345682',
        area: 'রামপুর',
        billMonth: '2024-01',
        billAmount: 750,
        dueDate: '2024-02-20',
        daysOverdue: 0,
        connectionType: 'residential',
        lastReminderSent: null,
        reminderCount: 0
      },
      {
        id: 5,
        customerName: 'করিম উদ্দিন',
        meterNumber: 'PB001239',
        phoneNumber: '01412345683',
        area: 'কালিগঞ্জ',
        billMonth: '2024-01',
        billAmount: 1200,
        dueDate: '2024-02-10',
        daysOverdue: 10,
        connectionType: 'commercial',
        lastReminderSent: '2024-02-08',
        reminderCount: 3
      }
    ];
    setPendingBills(mockPendingBills);
  }, []);

  // Filter bills
  const filteredBills = pendingBills.filter(bill => {
    const matchesSearch = bill.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         bill.meterNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         bill.phoneNumber.includes(searchTerm);
    const matchesArea = selectedArea === 'all' || bill.area === selectedArea;
    
    let matchesDueDate = true;
    if (dueDateFilter === 'overdue') {
      matchesDueDate = bill.daysOverdue > 0;
    } else if (dueDateFilter === 'due_soon') {
      matchesDueDate = bill.daysOverdue <= 0 && bill.daysOverdue >= -7;
    } else if (dueDateFilter === 'critical') {
      matchesDueDate = bill.daysOverdue > 7;
    }
    
    return matchesSearch && matchesArea && matchesDueDate;
  });

  // Calculate statistics
  const totalPending = pendingBills.length;
  const overdueBills = pendingBills.filter(bill => bill.daysOverdue > 0).length;
  const dueSoonBills = pendingBills.filter(bill => bill.daysOverdue <= 0 && bill.daysOverdue >= -7).length;
  const criticalBills = pendingBills.filter(bill => bill.daysOverdue > 7).length;
  const totalAmount = pendingBills.reduce((sum, bill) => sum + bill.billAmount, 0);
  const overdueAmount = pendingBills.filter(bill => bill.daysOverdue > 0).reduce((sum, bill) => sum + bill.billAmount, 0);

  // Get priority level
  const getPriorityLevel = (daysOverdue) => {
    if (daysOverdue > 7) return { level: 'critical', color: 'danger', text: 'জরুরি' };
    if (daysOverdue > 0) return { level: 'high', color: 'warning', text: 'উচ্চ' };
    if (daysOverdue >= -7) return { level: 'medium', color: 'info', text: 'মধ্যম' };
    return { level: 'low', color: 'success', text: 'কম' };
  };

  // Handle bill selection
  const handleBillSelection = (billId, isSelected) => {
    if (isSelected) {
      setSelectedBills([...selectedBills, billId]);
    } else {
      setSelectedBills(selectedBills.filter(id => id !== billId));
    }
  };

  // Select all bills
  const handleSelectAll = (isSelected) => {
    if (isSelected) {
      setSelectedBills(filteredBills.map(bill => bill.id));
    } else {
      setSelectedBills([]);
    }
  };

  // Send reminder
  const sendReminder = (e) => {
    e.preventDefault();
    
    const selectedBillsData = pendingBills.filter(bill => selectedBills.includes(bill.id));
    
    // Update reminder count and last sent date
    const updatedBills = pendingBills.map(bill => 
      selectedBills.includes(bill.id)
        ? {
            ...bill,
            lastReminderSent: new Date().toISOString().split('T')[0],
            reminderCount: bill.reminderCount + 1
          }
        : bill
    );
    
    setPendingBills(updatedBills);
    setShowAlert({ 
      show: true, 
      message: `${selectedBillsData.length}টি গ্রাহকের কাছে ${reminderData.method === 'sms' ? 'এসএমএস' : reminderData.method === 'call' ? 'ফোন কল' : 'ইমেইল'} পাঠানো হয়েছে।`, 
      type: 'success' 
    });
    
    setShowReminderModal(false);
    setSelectedBills([]);
    setTimeout(() => setShowAlert({ show: false, message: '', type: '' }), 3000);
  };

  // Export to CSV
  const exportToCSV = () => {
    const csvContent = [
      ['গ্রাহকের নাম', 'মিটার নম্বর', 'ফোন নম্বর', 'এলাকা', 'বিলের মাস', 'বিলের পরিমাণ', 'শেষ তারিখ', 'বকেয়া দিন', 'অগ্রাধিকার'],
      ...filteredBills.map(bill => [
        bill.customerName,
        bill.meterNumber,
        bill.phoneNumber,
        bill.area,
        new Date(bill.billMonth).toLocaleDateString('bn-BD', { year: 'numeric', month: 'long' }),
        bill.billAmount,
        new Date(bill.dueDate).toLocaleDateString('bn-BD'),
        bill.daysOverdue > 0 ? `${bill.daysOverdue} দিন বকেয়া` : `${Math.abs(bill.daysOverdue)} দিন বাকি`,
        getPriorityLevel(bill.daysOverdue).text
      ])
    ].map(row => row.join(',')).join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'pending_bills.csv';
    link.click();
  };

  return (
    <Container fluid>
      <Row className="mb-4">
        <Col>
          <h2>পল্লী বিদ্যুৎ - বকেয়া বিল ব্যবস্থাপনা</h2>
        </Col>
      </Row>

      {showAlert.show && (
        <Alert variant={showAlert.type} className="mb-3">
          {showAlert.message}
        </Alert>
      )}

      {/* Summary Cards */}
      <Row className="mb-4">
        <Col md={3}>
          <Card className="summary-card total">
            <Card.Body className="text-center">
              <h3>{totalPending}</h3>
              <p>মোট বকেয়া</p>
              <small>৳{totalAmount.toLocaleString()}</small>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card overdue">
            <Card.Body className="text-center">
              <h3>{overdueBills}</h3>
              <p>মেয়াদোত্তীর্ণ</p>
              <small>৳{overdueAmount.toLocaleString()}</small>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card due-soon">
            <Card.Body className="text-center">
              <h3>{dueSoonBills}</h3>
              <p>শীঘ্রই মেয়াদোত্তীর্ণ</p>
            </Card.Body>
          </Card>
        </Col>
        <Col md={3}>
          <Card className="summary-card critical">
            <Card.Body className="text-center">
              <h3>{criticalBills}</h3>
              <p>জরুরি</p>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Collection Progress */}
      <Row className="mb-4">
        <Col>
          <Card>
            <Card.Body>
              <h5>সংগ্রহের অগ্রগতি</h5>
              <div className="mb-2">
                <div className="d-flex justify-content-between">
                  <span>বকেয়া: ৳{overdueAmount.toLocaleString()}</span>
                  <span>মোট: ৳{totalAmount.toLocaleString()}</span>
                </div>
                <ProgressBar 
                  variant="danger" 
                  now={(overdueAmount / totalAmount) * 100} 
                  className="mt-2"
                />
              </div>
              <small className="text-muted">
                {((overdueAmount / totalAmount) * 100).toFixed(1)}% বিল বকেয়া রয়েছে
              </small>
            </Card.Body>
          </Card>
        </Col>
      </Row>

      {/* Filters and Actions */}
      <Row className="mb-3">
        <Col md={3}>
          <InputGroup>
            <Form.Control
              type="text"
              placeholder="গ্রাহক খুঁজুন..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            <Button variant="outline-secondary">
              <FaSearch />
            </Button>
          </InputGroup>
        </Col>
        <Col md={2}>
          <Form.Select value={selectedArea} onChange={(e) => setSelectedArea(e.target.value)}>
            <option value="all">সব এলাকা</option>
            <option value="রামপুর">রামপুর</option>
            <option value="কালিগঞ্জ">কালিগঞ্জ</option>
            <option value="বাজারপাড়া">বাজারপাড়া</option>
          </Form.Select>
        </Col>
        <Col md={2}>
          <Form.Select value={dueDateFilter} onChange={(e) => setDueDateFilter(e.target.value)}>
            <option value="all">সব বিল</option>
            <option value="overdue">মেয়াদোত্তীর্ণ</option>
            <option value="due_soon">শীঘ্রই মেয়াদোত্তীর্ণ</option>
            <option value="critical">জরুরি</option>
          </Form.Select>
        </Col>
        <Col md={5} className="text-end">
          <Button 
            variant="warning" 
            className="me-2" 
            disabled={selectedBills.length === 0}
            onClick={() => setShowReminderModal(true)}
          >
            <FaSms className="me-1" /> রিমাইন্ডার পাঠান ({selectedBills.length})
          </Button>
          <Button variant="success" className="me-2" onClick={exportToCSV}>
            <FaFileExport className="me-1" /> এক্সপোর্ট
          </Button>
          <Button variant="info" onClick={() => window.print()}>
            <FaPrint className="me-1" /> প্রিন্ট
          </Button>
        </Col>
      </Row>

      {/* Bills Table */}
      <Card>
        <Card.Body>
          <div className="d-flex justify-content-between align-items-center mb-3">
            <h5>বকেয়া বিলের তালিকা</h5>
            <Form.Check
              type="checkbox"
              label="সব নির্বাচন করুন"
              checked={selectedBills.length === filteredBills.length && filteredBills.length > 0}
              onChange={(e) => handleSelectAll(e.target.checked)}
            />
          </div>
          
          <Table responsive striped hover>
            <thead>
              <tr>
                <th>নির্বাচন</th>
                <th>গ্রাহকের তথ্য</th>
                <th>বিলের তথ্য</th>
                <th>শেষ তারিখ</th>
                <th>অগ্রাধিকার</th>
                <th>রিমাইন্ডার</th>
                <th>কার্যক্রম</th>
              </tr>
            </thead>
            <tbody>
              {filteredBills.map(bill => {
                const priority = getPriorityLevel(bill.daysOverdue);
                
                return (
                  <tr key={bill.id} className={priority.level === 'critical' ? 'table-danger' : ''}>
                    <td>
                      <Form.Check
                        type="checkbox"
                        checked={selectedBills.includes(bill.id)}
                        onChange={(e) => handleBillSelection(bill.id, e.target.checked)}
                      />
                    </td>
                    <td>
                      <strong>{bill.customerName}</strong>
                      <br />
                      <small className="text-muted">{bill.meterNumber}</small>
                      <br />
                      <small className="text-muted">{bill.phoneNumber}</small>
                      <br />
                      <small className="text-muted">{bill.area}</small>
                    </td>
                    <td>
                      <div>
                        <strong className="text-primary">৳{bill.billAmount}</strong>
                        <br />
                        <small>
                          {new Date(bill.billMonth).toLocaleDateString('bn-BD', { 
                            year: 'numeric', 
                            month: 'long' 
                          })}
                        </small>
                        <br />
                        <Badge bg={bill.connectionType === 'residential' ? 'primary' : 'warning'}>
                          {bill.connectionType === 'residential' ? 'আবাসিক' : 'বাণিজ্যিক'}
                        </Badge>
                      </div>
                    </td>
                    <td>
                      <div className={bill.daysOverdue > 0 ? 'text-danger' : 'text-warning'}>
                        {new Date(bill.dueDate).toLocaleDateString('bn-BD')}
                        <br />
                        <small>
                          {bill.daysOverdue > 0 
                            ? `${bill.daysOverdue} দিন বকেয়া` 
                            : `${Math.abs(bill.daysOverdue)} দিন বাকি`
                          }
                        </small>
                      </div>
                    </td>
                    <td>
                      <Badge bg={priority.color}>
                        {priority.text}
                      </Badge>
                      {priority.level === 'critical' && (
                        <div className="mt-1">
                          <FaExclamationTriangle className="text-danger" />
                        </div>
                      )}
                    </td>
                    <td>
                      <div>
                        <small>পাঠানো: {bill.reminderCount}টি</small>
                        <br />
                        {bill.lastReminderSent && (
                          <small className="text-muted">
                            শেষ: {new Date(bill.lastReminderSent).toLocaleDateString('bn-BD')}
                          </small>
                        )}
                      </div>
                    </td>
                    <td>
                      <div className="d-flex flex-column gap-1">
                        <Button variant="outline-warning" size="sm">
                          <FaPhone className="me-1" /> কল
                        </Button>
                        <Button variant="outline-info" size="sm">
                          <FaSms className="me-1" /> এসএমএস
                        </Button>
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </Table>

          {filteredBills.length === 0 && (
            <div className="text-center py-4">
              <p className="text-muted">কোন বকেয়া বিল পাওয়া যায়নি।</p>
            </div>
          )}
        </Card.Body>
      </Card>

      {/* Reminder Modal */}
      <Modal show={showReminderModal} onHide={() => setShowReminderModal(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>রিমাইন্ডার পাঠান</Modal.Title>
        </Modal.Header>
        <Form onSubmit={sendReminder}>
          <Modal.Body>
            <div className="mb-3">
              <h6>নির্বাচিত গ্রাহক: {selectedBills.length}টি</h6>
              <div className="selected-customers">
                {pendingBills
                  .filter(bill => selectedBills.includes(bill.id))
                  .map(bill => (
                    <Badge key={bill.id} bg="secondary" className="me-1 mb-1">
                      {bill.customerName}
                    </Badge>
                  ))
                }
              </div>
            </div>
            
            <Row>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>রিমাইন্ডারের পদ্ধতি</Form.Label>
                  <Form.Select
                    value={reminderData.method}
                    onChange={(e) => setReminderData({...reminderData, method: e.target.value})}
                  >
                    <option value="sms">এসএমএস</option>
                    <option value="call">ফোন কল</option>
                    <option value="email">ইমেইল</option>
                  </Form.Select>
                </Form.Group>
              </Col>
              <Col md={6}>
                <Form.Group className="mb-3">
                  <Form.Label>প্রেরণের সময়</Form.Label>
                  <Form.Select>
                    <option value="now">এখনই</option>
                    <option value="morning">সকাল ৯টায়</option>
                    <option value="afternoon">দুপুর ২টায়</option>
                    <option value="evening">সন্ধ্যা ৬টায়</option>
                  </Form.Select>
                </Form.Group>
              </Col>
            </Row>
            
            <Form.Group className="mb-3">
              <Form.Label>বার্তা</Form.Label>
              <Form.Control
                as="textarea"
                rows={4}
                value={reminderData.message}
                onChange={(e) => setReminderData({...reminderData, message: e.target.value})}
                placeholder="রিমাইন্ডার বার্তা লিখুন..."
              />
              <Form.Text className="text-muted">
                বার্তার দৈর্ঘ্য: {reminderData.message.length}/160 অক্ষর
              </Form.Text>
            </Form.Group>
            
            <div className="reminder-templates">
              <h6>টেমপ্লেট বার্তা:</h6>
              <div className="d-flex flex-wrap gap-2">
                <Button 
                  variant="outline-secondary" 
                  size="sm"
                  onClick={() => setReminderData({
                    ...reminderData, 
                    message: 'আপনার বিদ্যুৎ বিলের পরিশোধের শেষ তারিখ আসন্ন। অনুগ্রহ করে যথাসময়ে পরিশোধ করুন।'
                  })}
                >
                  সাধারণ রিমাইন্ডার
                </Button>
                <Button 
                  variant="outline-warning" 
                  size="sm"
                  onClick={() => setReminderData({
                    ...reminderData, 
                    message: 'আপনার বিদ্যুৎ বিল বকেয়া রয়েছে। অবিলম্বে পরিশোধ করুন, অন্যথায় সংযোগ বিচ্ছিন্ন হতে পারে।'
                  })}
                >
                  জরুরি নোটিশ
                </Button>
                <Button 
                  variant="outline-info" 
                  size="sm"
                  onClick={() => setReminderData({
                    ...reminderData, 
                    message: 'আপনার বিদ্যুৎ বিল পরিশোধের জন্য ধন্যবাদ। আমাদের অফিসে এসে বিল পরিশোধ করুন।'
                  })}
                >
                  ভদ্র অনুরোধ
                </Button>
              </div>
            </div>
          </Modal.Body>
          <Modal.Footer>
            <Button variant="secondary" onClick={() => setShowReminderModal(false)}>
              বাতিল
            </Button>
            <Button variant="warning" type="submit">
              <FaSms className="me-1" /> রিমাইন্ডার পাঠান
            </Button>
          </Modal.Footer>
        </Form>
      </Modal>
    </Container>
  );
};

export default PendingBills;