import React, { useState } from 'react';
import { Card, Row, Col, Form, Button } from 'react-bootstrap';
import { 
  Chart as ChartJS, 
  CategoryScale, 
  LinearScale, 
  PointElement, 
  LineElement, 
  BarElement,
  Title, 
  Tooltip, 
  Legend, 
  ArcElement 
} from 'chart.js';
import { Line, Bar, Pie } from 'react-chartjs-2';
import './Dashboard.css';

// Register ChartJS components
ChartJS.register(
  CategoryScale, 
  LinearScale, 
  PointElement, 
  LineElement, 
  BarElement,
  Title, 
  Tooltip, 
  Legend,
  ArcElement
);

const Dashboard = () => {
  // State for date filters
  const [dateRange, setDateRange] = useState({
    startDate: new Date(new Date().setDate(new Date().getDate() - 30)).toISOString().split('T')[0],
    endDate: new Date().toISOString().split('T')[0]
  });

  // Mock data for sales chart
  const salesData = {
    labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
    datasets: [
      {
        label: 'Sales',
        data: [12500, 19200, 15000, 22500, 18300, 24100, 27800],
        borderColor: 'rgb(53, 162, 235)',
        backgroundColor: 'rgba(53, 162, 235, 0.5)',
      },
    ],
  };

  // Mock data for top products
  const topProductsData = {
    labels: ['Product A', 'Product B', 'Product C', 'Product D', 'Product E'],
    datasets: [
      {
        label: 'Units Sold',
        data: [120, 98, 85, 72, 63],
        backgroundColor: 'rgba(75, 192, 192, 0.6)',
      },
    ],
  };

  // Mock data for payment methods
  const paymentMethodsData = {
    labels: ['Cash', 'Card', 'Mobile Banking', 'Bank Transfer'],
    datasets: [
      {
        label: 'Payment Methods',
        data: [45, 25, 20, 10],
        backgroundColor: [
          'rgba(255, 99, 132, 0.6)',
          'rgba(54, 162, 235, 0.6)',
          'rgba(255, 206, 86, 0.6)',
          'rgba(75, 192, 192, 0.6)',
        ],
        borderColor: [
          'rgba(255, 99, 132, 1)',
          'rgba(54, 162, 235, 1)',
          'rgba(255, 206, 86, 1)',
          'rgba(75, 192, 192, 1)',
        ],
        borderWidth: 1,
      },
    ],
  };

  // Handle date range change
  const handleDateChange = (e) => {
    const { name, value } = e.target;
    setDateRange({
      ...dateRange,
      [name]: value
    });
  };

  // Chart options
  const lineOptions = {
    responsive: true,
    plugins: {
      legend: {
        position: 'top',
      },
      title: {
        display: true,
        text: 'Sales Trend',
      },
    },
  };

  const barOptions = {
    responsive: true,
    plugins: {
      legend: {
        position: 'top',
      },
      title: {
        display: true,
        text: 'Top Selling Products',
      },
    },
  };

  const pieOptions = {
    responsive: true,
    plugins: {
      legend: {
        position: 'right',
      },
      title: {
        display: true,
        text: 'Payment Methods',
      },
    },
  };

  return (
    <div className="dashboard-container">
      <Card className="mb-4">
        <Card.Header>
          <h4 className="mb-0">Sales Dashboard</h4>
        </Card.Header>
        <Card.Body>
          <Form className="mb-4">
            <Row>
              <Col md={4}>
                <Form.Group>
                  <Form.Label>Start Date</Form.Label>
                  <Form.Control
                    type="date"
                    name="startDate"
                    value={dateRange.startDate}
                    onChange={handleDateChange}
                  />
                </Form.Group>
              </Col>
              <Col md={4}>
                <Form.Group>
                  <Form.Label>End Date</Form.Label>
                  <Form.Control
                    type="date"
                    name="endDate"
                    value={dateRange.endDate}
                    onChange={handleDateChange}
                  />
                </Form.Group>
              </Col>
              <Col md={4} className="d-flex align-items-end">
                <Button variant="primary" className="w-100">
                  Apply Filter
                </Button>
              </Col>
            </Row>
          </Form>

          <Row className="mb-4">
            <Col md={3}>
              <Card className="summary-card bg-primary text-white">
                <Card.Body>
                  <h6>Total Sales</h6>
                  <h3>৳139,400</h3>
                  <div className="small">+12.5% from last month</div>
                </Card.Body>
              </Card>
            </Col>
            <Col md={3}>
              <Card className="summary-card bg-success text-white">
                <Card.Body>
                  <h6>Total Orders</h6>
                  <h3>428</h3>
                  <div className="small">+8.2% from last month</div>
                </Card.Body>
              </Card>
            </Col>
            <Col md={3}>
              <Card className="summary-card bg-info text-white">
                <Card.Body>
                  <h6>Average Order</h6>
                  <h3>৳325.70</h3>
                  <div className="small">+3.8% from last month</div>
                </Card.Body>
              </Card>
            </Col>
            <Col md={3}>
              <Card className="summary-card bg-warning text-white">
                <Card.Body>
                  <h6>Low Stock Items</h6>
                  <h3>12</h3>
                  <div className="small">Action needed</div>
                </Card.Body>
              </Card>
            </Col>
          </Row>

          <Row>
            <Col lg={8} className="mb-4">
              <Card>
                <Card.Body>
                  <Line options={lineOptions} data={salesData} />
                </Card.Body>
              </Card>
            </Col>
            <Col lg={4} className="mb-4">
              <Card>
                <Card.Body>
                  <Pie options={pieOptions} data={paymentMethodsData} />
                </Card.Body>
              </Card>
            </Col>
          </Row>

          <Row>
            <Col md={12}>
              <Card>
                <Card.Body>
                  <Bar options={barOptions} data={topProductsData} />
                </Card.Body>
              </Card>
            </Col>
          </Row>
        </Card.Body>
      </Card>

      <Card>
        <Card.Header>
          <h4 className="mb-0">Inventory Status</h4>
        </Card.Header>
        <Card.Body>
          <Row>
            <Col md={12}>
              <table className="table table-bordered table-hover">
                <thead>
                  <tr>
                    <th>Product</th>
                    <th>SKU</th>
                    <th>Category</th>
                    <th>Current Stock</th>
                    <th>Status</th>
                  </tr>
                </thead>
                <tbody>
                  <tr>
                    <td>Product A</td>
                    <td>SKU001</td>
                    <td>Electronics</td>
                    <td>5</td>
                    <td><span className="badge bg-warning">Low Stock</span></td>
                  </tr>
                  <tr>
                    <td>Product B</td>
                    <td>SKU002</td>
                    <td>Clothing</td>
                    <td>0</td>
                    <td><span className="badge bg-danger">Out of Stock</span></td>
                  </tr>
                  <tr>
                    <td>Product C</td>
                    <td>SKU003</td>
                    <td>Food</td>
                    <td>3</td>
                    <td><span className="badge bg-warning">Low Stock</span></td>
                  </tr>
                  <tr>
                    <td>Product D</td>
                    <td>SKU004</td>
                    <td>Electronics</td>
                    <td>2</td>
                    <td><span className="badge bg-warning">Low Stock</span></td>
                  </tr>
                  <tr>
                    <td>Product E</td>
                    <td>SKU005</td>
                    <td>Clothing</td>
                    <td>15</td>
                    <td><span className="badge bg-success">In Stock</span></td>
                  </tr>
                </tbody>
              </table>
            </Col>
          </Row>
        </Card.Body>
      </Card>
    </div>
  );
};

export default Dashboard;