import React, { useState, useEffect, useRef } from 'react';
import { Row, Col, Card, Button, Form, Table, Modal, InputGroup } from 'react-bootstrap';
import { FaBarcode, FaPlus, FaMinus, FaTrash, FaPrint, FaSave, FaMoneyBill, FaMobileAlt } from 'react-icons/fa';
import DSOStockSelectionModal from '../../components/DSOStockSelectionModal';
import DSOAllocationBadge from '../../components/DSOAllocationBadge';
import './POS.css';

const POS = () => {
  // State for cart items
  const [cart, setCart] = useState([]);
  const [products, setProducts] = useState([]);
  const [filteredProducts, setFilteredProducts] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [categories, setCategories] = useState([]);
  const [selectedCategory, setSelectedCategory] = useState('all');
  
  // State for checkout
  const [subTotal, setSubTotal] = useState(0);
  const [discount, setDiscount] = useState(0);
  const [tax, setTax] = useState(0);
  const [total, setTotal] = useState(0);
  const [showCheckout, setShowCheckout] = useState(false);
  const [paymentMethod, setPaymentMethod] = useState('cash');
  const [mobileNumber, setMobileNumber] = useState('');
  const [customerName, setCustomerName] = useState('');
  const [amountPaid, setAmountPaid] = useState(0);
  const [change, setChange] = useState(0);
  const [transactionId, setTransactionId] = useState('');
  
  // DSO-related state
  const [showDSOStockModal, setShowDSOStockModal] = useState(false);
  const [dsoStockSelections, setDsoStockSelections] = useState({});
  
  // Barcode scanner ref
  const barcodeInputRef = useRef(null);
  
  // Mock data for products (in a real app, this would come from an API)
  useEffect(() => {
    // Simulating API call to fetch products
    const mockProducts = [
      { id: 1, name: 'Coca Cola', price: 35, category: 'beverages', barcode: '8901234567890', stock: 50, image: 'https://via.placeholder.com/50' },
      { id: 2, name: 'Lays Chips', price: 25, category: 'snacks', barcode: '8901234567891', stock: 30, image: 'https://via.placeholder.com/50' },
      { id: 3, name: 'Dairy Milk', price: 60, category: 'chocolates', barcode: '8901234567892', stock: 25, image: 'https://via.placeholder.com/50' },
      { id: 4, name: 'Bread', price: 40, category: 'bakery', barcode: '8901234567893', stock: 15, image: 'https://via.placeholder.com/50' },
      { id: 5, name: 'Eggs (6pcs)', price: 42, category: 'dairy', barcode: '8901234567894', stock: 20, image: 'https://via.placeholder.com/50' },
      { id: 6, name: 'Milk 500ml', price: 30, category: 'dairy', barcode: '8901234567895', stock: 18, image: 'https://via.placeholder.com/50' },
      { id: 7, name: 'Pepsi', price: 35, category: 'beverages', barcode: '8901234567896', stock: 45, image: 'https://via.placeholder.com/50' },
      { id: 8, name: 'Biscuits', price: 20, category: 'snacks', barcode: '8901234567897', stock: 40, image: 'https://via.placeholder.com/50' },
      { id: 9, name: 'Rice 1kg', price: 65, category: 'groceries', barcode: '8901234567898', stock: 25, image: 'https://via.placeholder.com/50' },
      { id: 10, name: 'Sugar 1kg', price: 50, category: 'groceries', barcode: '8901234567899', stock: 30, image: 'https://via.placeholder.com/50' },
      { id: 11, name: 'Tea 250g', price: 120, category: 'groceries', barcode: '8901234567810', stock: 15, image: 'https://via.placeholder.com/50' },
      { id: 12, name: 'Soap', price: 35, category: 'toiletries', barcode: '8901234567811', stock: 35, image: 'https://via.placeholder.com/50' },
      // DSO Products
      { id: 13, name: 'Grameenphone SIM Card', price: 50, category: 'dso', barcode: '8901234567812', stock: 100, image: 'https://via.placeholder.com/50' },
      { id: 14, name: 'Banglalink SIM Card', price: 45, category: 'dso', barcode: '8901234567813', stock: 80, image: 'https://via.placeholder.com/50' },
      { id: 15, name: 'Robi SIM Card', price: 48, category: 'dso', barcode: '8901234567814', stock: 90, image: 'https://via.placeholder.com/50' },
      { id: 16, name: 'Airtel SIM Card', price: 40, category: 'dso', barcode: '8901234567815', stock: 70, image: 'https://via.placeholder.com/50' },
      { id: 17, name: 'SIM Starter Kit', price: 25, category: 'dso', barcode: '8901234567816', stock: 50, image: 'https://via.placeholder.com/50' },
      { id: 18, name: 'Data Card 1GB', price: 100, category: 'dso', barcode: '8901234567817', stock: 60, image: 'https://via.placeholder.com/50' },
      { id: 19, name: 'Data Card 5GB', price: 300, category: 'dso', barcode: '8901234567818', stock: 40, image: 'https://via.placeholder.com/50' },
      { id: 20, name: 'Voice Card 100min', price: 80, category: 'dso', barcode: '8901234567819', stock: 45, image: 'https://via.placeholder.com/50' },
    ];
    
    setProducts(mockProducts);
    setFilteredProducts(mockProducts);
    
    // Extract unique categories
    const uniqueCategories = [...new Set(mockProducts.map(product => product.category))];
    setCategories(uniqueCategories);
    
    // Focus on barcode input
    if (barcodeInputRef.current) {
      barcodeInputRef.current.focus();
    }
  }, []);
  
  // Calculate totals whenever cart changes
  useEffect(() => {
    const calculatedSubTotal = cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
    setSubTotal(calculatedSubTotal);
    
    const calculatedTotal = calculatedSubTotal - discount + tax;
    setTotal(calculatedTotal);
    
    // Calculate change if amount paid is set
    if (amountPaid > 0) {
      setChange(amountPaid - calculatedTotal);
    }
  }, [cart, discount, tax, amountPaid]);
  
  // Filter products based on search and category
  useEffect(() => {
    let filtered = products;
    
    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(product => 
        product.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        product.barcode.includes(searchTerm)
      );
    }
    
    // Filter by category
    if (selectedCategory !== 'all') {
      filtered = filtered.filter(product => product.category === selectedCategory);
    }
    
    setFilteredProducts(filtered);
  }, [searchTerm, selectedCategory, products]);
  
  // Handle barcode scan
  const handleBarcodeSubmit = (e) => {
    e.preventDefault();
    const barcode = e.target.barcode.value;
    
    // Find product by barcode
    const product = products.find(p => p.barcode === barcode);
    if (product) {
      addToCart(product);
    }
    
    // Clear barcode input
    e.target.barcode.value = '';
  };
  
  // Add product to cart
  const addToCart = (product) => {
    // Check if product is already in cart
    const existingItemIndex = cart.findIndex(item => item.id === product.id);
    
    if (existingItemIndex !== -1) {
      // Update quantity if product already in cart
      const updatedCart = [...cart];
      updatedCart[existingItemIndex].quantity += 1;
      setCart(updatedCart);
    } else {
      // Add new product to cart
      setCart([...cart, { ...product, quantity: 1 }]);
    }
  };
  
  // Update item quantity in cart
  const updateQuantity = (id, newQuantity) => {
    if (newQuantity < 1) return;
    
    const updatedCart = cart.map(item => 
      item.id === id ? { ...item, quantity: newQuantity } : item
    );
    
    setCart(updatedCart);
  };
  
  // Remove item from cart
  const removeFromCart = (id) => {
    const updatedCart = cart.filter(item => item.id !== id);
    setCart(updatedCart);
  };
  
  // Clear the entire cart
  const clearCart = () => {
    setCart([]);
    setDiscount(0);
    setTax(0);
    setCustomerName('');
    setMobileNumber('');
  };
  
  // Handle checkout process
  // DSO Stock Selection Functions
  const handleInitiateCheckout = () => {
    // Check if cart contains DSO items
    const hasDSOItems = cart.some(item => {
      const name = item.name.toLowerCase();
      return name.includes('card') || name.includes('sim') || name.includes('kit');
    });

    if (hasDSOItems) {
      // Show DSO stock selection modal first
      setShowDSOStockModal(true);
    } else {
      // Proceed directly to checkout
      setShowCheckout(true);
    }
  };

  const handleDSOStockSelectionComplete = (stockSelections) => {
    setDsoStockSelections(stockSelections);
    setShowDSOStockModal(false);
    setShowCheckout(true);
  };

  const handleCheckout = async () => {
    try {
      // Prepare sale data
      const saleData = {
        items: cart,
        subTotal,
        discount,
        tax,
        total,
        paymentMethod,
        customerName,
        mobileNumber,
        amountPaid,
        change,
        transactionId,
        dsoStockSelections,
        date: new Date()
      };

      // In a real app, this would send the sale to the backend
      console.log('Sale Data:', saleData);

      // If there are DSO items with allocated stock selections, record the sale
      if (Object.keys(dsoStockSelections).length > 0) {
        await recordDSOSale(saleData);
      }

      // Reset the POS
      clearCart();
      setShowCheckout(false);
      setShowDSOStockModal(false);
      setAmountPaid(0);
      setChange(0);
      setPaymentMethod('cash');
      setCustomerName('');
      setMobileNumber('');
      setTransactionId('');
      setDsoStockSelections({});

      // Show success message
      alert('Sale completed successfully!');
    } catch (error) {
      console.error('Error completing sale:', error);
      alert('Error completing sale. Please try again.');
    }
  };

  const recordDSOSale = async (saleData) => {
    try {
      // Calculate DSO sales from stock selections
      const dsoSales = {
        cardsSold: 0,
        cardTypesSold: [],
        simsSold: 0,
        kitsSold: 0
      };

      Object.entries(saleData.dsoStockSelections).forEach(([itemId, selection]) => {
        if (selection.useAllocated) {
          const item = cart.find(cartItem => cartItem.id.toString() === itemId);
          if (item) {
            if (selection.itemType === 'cards') {
              dsoSales.cardsSold += selection.quantity;
              dsoSales.cardTypesSold.push({
                type: item.name,
                quantity: selection.quantity
              });
            } else if (selection.itemType === 'sims') {
              dsoSales.simsSold += selection.quantity;
            } else if (selection.itemType === 'kits') {
              dsoSales.kitsSold += selection.quantity;
            }
          }
        }
      });

      // Only record if there are actual DSO sales from allocated stock
      if (dsoSales.cardsSold > 0 || dsoSales.simsSold > 0 || dsoSales.kitsSold > 0) {
        const response = await fetch('/api/dso/record-sale', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            ...dsoSales,
            saleAmount: saleData.total,
            customerName: saleData.customerName,
            paymentMethod: saleData.paymentMethod,
            transactionId: saleData.transactionId,
            notes: `POS Sale - ${saleData.items.length} items`
          }),
        });

        if (!response.ok) {
          throw new Error('Failed to record DSO sale');
        }
      }
    } catch (error) {
      console.error('Error recording DSO sale:', error);
      throw error;
    }
  };
  
  return (
    <div className="pos-container">
      <Row>
        {/* Left Side - Products */}
        <Col md={7}>
          <Card className="mb-3">
            <Card.Header className="d-flex justify-content-between align-items-center">
              <h5 className="mb-0">Products</h5>
              <Form className="d-flex" style={{ width: '60%' }}>
                <Form.Control
                  type="search"
                  placeholder="Search products..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </Form>
            </Card.Header>
            <Card.Body>
              {/* Category Filters */}
              <div className="category-filters mb-3">
                <Button 
                  variant={selectedCategory === 'all' ? 'primary' : 'outline-primary'} 
                  className="me-2 mb-2"
                  onClick={() => setSelectedCategory('all')}
                >
                  All
                </Button>
                {categories.map(category => (
                  <Button 
                    key={category}
                    variant={selectedCategory === category ? 'primary' : 'outline-primary'} 
                    className="me-2 mb-2"
                    onClick={() => setSelectedCategory(category)}
                  >
                    {category.charAt(0).toUpperCase() + category.slice(1)}
                  </Button>
                ))}
              </div>
              
              {/* Products Grid */}
              <Row className="product-grid">
                {filteredProducts.map(product => (
                  <Col key={product.id} xs={6} md={4} lg={3} className="mb-3">
                    <Card 
                      className="product-card h-100" 
                      onClick={() => addToCart(product)}
                    >
                      <Card.Img variant="top" src={product.image} className="product-image" />
                      <Card.Body className="p-2">
                        <Card.Title className="product-name">{product.name}</Card.Title>
                        <Card.Text className="product-price">৳{product.price}</Card.Text>
                      </Card.Body>
                    </Card>
                  </Col>
                ))}
              </Row>
            </Card.Body>
          </Card>
        </Col>
        
        {/* Right Side - Cart & Checkout */}
        <Col md={5}>
          {/* DSO Allocation Badge */}
          <DSOAllocationBadge />
          
          <Card className="mb-3">
            <Card.Header className="d-flex justify-content-between align-items-center">
              <h5 className="mb-0">Current Sale</h5>
              <Form onSubmit={handleBarcodeSubmit} className="d-flex">
                <InputGroup>
                  <Form.Control
                    name="barcode"
                    placeholder="Scan barcode..."
                    ref={barcodeInputRef}
                  />
                  <Button type="submit" variant="primary">
                    <FaBarcode />
                  </Button>
                </InputGroup>
              </Form>
            </Card.Header>
            <Card.Body className="cart-container">
              {/* Cart Items */}
              <div className="cart-items">
                <Table responsive>
                  <thead>
                    <tr>
                      <th>Item</th>
                      <th>Price</th>
                      <th>Qty</th>
                      <th>Total</th>
                      <th></th>
                    </tr>
                  </thead>
                  <tbody>
                    {cart.length === 0 ? (
                      <tr>
                        <td colSpan="5" className="text-center py-3">
                          Cart is empty
                        </td>
                      </tr>
                    ) : (
                      cart.map(item => (
                        <tr key={item.id}>
                          <td>{item.name}</td>
                          <td>৳{item.price}</td>
                          <td>
                            <div className="quantity-control">
                              <Button 
                                size="sm" 
                                variant="outline-secondary"
                                onClick={() => updateQuantity(item.id, item.quantity - 1)}
                              >
                                <FaMinus />
                              </Button>
                              <span className="mx-2">{item.quantity}</span>
                              <Button 
                                size="sm" 
                                variant="outline-secondary"
                                onClick={() => updateQuantity(item.id, item.quantity + 1)}
                              >
                                <FaPlus />
                              </Button>
                            </div>
                          </td>
                          <td>৳{(item.price * item.quantity).toFixed(2)}</td>
                          <td>
                            <Button 
                              size="sm" 
                              variant="danger"
                              onClick={() => removeFromCart(item.id)}
                            >
                              <FaTrash />
                            </Button>
                          </td>
                        </tr>
                      ))
                    )}
                  </tbody>
                </Table>
              </div>
              
              {/* Cart Summary */}
              <div className="cart-summary mt-3">
                <div className="d-flex justify-content-between mb-2">
                  <span>Subtotal:</span>
                  <span>৳{subTotal.toFixed(2)}</span>
                </div>
                <div className="d-flex justify-content-between mb-2">
                  <span>Discount:</span>
                  <InputGroup size="sm">
                    <Form.Control
                      type="number"
                      value={discount}
                      onChange={(e) => setDiscount(Number(e.target.value))}
                      min="0"
                    />
                    <InputGroup.Text>৳</InputGroup.Text>
                  </InputGroup>
                </div>
                <div className="d-flex justify-content-between mb-2">
                  <span>Tax:</span>
                  <InputGroup size="sm">
                    <Form.Control
                      type="number"
                      value={tax}
                      onChange={(e) => setTax(Number(e.target.value))}
                      min="0"
                    />
                    <InputGroup.Text>৳</InputGroup.Text>
                  </InputGroup>
                </div>
                <div className="d-flex justify-content-between mb-2 total-row">
                  <span className="fw-bold">Total:</span>
                  <span className="fw-bold">৳{total.toFixed(2)}</span>
                </div>
              </div>
              
              {/* Action Buttons */}
              <div className="cart-actions mt-3">
                <Row>
                  <Col>
                    <Button 
                      variant="danger" 
                      className="w-100 mb-2"
                      onClick={clearCart}
                      disabled={cart.length === 0}
                    >
                      Cancel
                    </Button>
                  </Col>
                  <Col>
                    <Button 
                      variant="success" 
                      className="w-100 mb-2"
                      onClick={handleInitiateCheckout}
                      disabled={cart.length === 0}
                    >
                      Checkout
                    </Button>
                  </Col>
                </Row>
              </div>
            </Card.Body>
          </Card>
        </Col>
      </Row>
      
      {/* Checkout Modal */}
      <Modal show={showCheckout} onHide={() => setShowCheckout(false)} size="lg">
        <Modal.Header closeButton>
          <Modal.Title>Complete Sale</Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <Row>
            <Col md={7}>
              <h5>Order Summary</h5>
              <Table responsive>
                <thead>
                  <tr>
                    <th>Item</th>
                    <th>Qty</th>
                    <th>Price</th>
                    <th>Total</th>
                  </tr>
                </thead>
                <tbody>
                  {cart.map(item => (
                    <tr key={item.id}>
                      <td>{item.name}</td>
                      <td>{item.quantity}</td>
                      <td>৳{item.price}</td>
                      <td>৳{(item.price * item.quantity).toFixed(2)}</td>
                    </tr>
                  ))}
                </tbody>
                <tfoot>
                  <tr>
                    <td colSpan="3" className="text-end">Subtotal:</td>
                    <td>৳{subTotal.toFixed(2)}</td>
                  </tr>
                  <tr>
                    <td colSpan="3" className="text-end">Discount:</td>
                    <td>৳{discount.toFixed(2)}</td>
                  </tr>
                  <tr>
                    <td colSpan="3" className="text-end">Tax:</td>
                    <td>৳{tax.toFixed(2)}</td>
                  </tr>
                  <tr>
                    <td colSpan="3" className="text-end fw-bold">Total:</td>
                    <td className="fw-bold">৳{total.toFixed(2)}</td>
                  </tr>
                </tfoot>
              </Table>
            </Col>
            <Col md={5}>
              <h5>Payment Details</h5>
              <Form>
                <Form.Group className="mb-3">
                  <Form.Label>Customer Name</Form.Label>
                  <Form.Control 
                    type="text" 
                    value={customerName}
                    onChange={(e) => setCustomerName(e.target.value)}
                    placeholder="Enter customer name"
                  />
                </Form.Group>
                
                <Form.Group className="mb-3">
                  <Form.Label>Mobile Number</Form.Label>
                  <Form.Control 
                    type="text" 
                    value={mobileNumber}
                    onChange={(e) => setMobileNumber(e.target.value)}
                    placeholder="Enter mobile number"
                  />
                </Form.Group>
                
                <Form.Group className="mb-3">
                  <Form.Label>Payment Method</Form.Label>
                  <div>
                    <Form.Check
                      inline
                      type="radio"
                      label={<><FaMoneyBill className="me-1" /> Cash</>}
                      name="paymentMethod"
                      id="cash"
                      checked={paymentMethod === 'cash'}
                      onChange={() => setPaymentMethod('cash')}
                    />
                    <Form.Check
                      inline
                      type="radio"
                      label={<><FaMobileAlt className="me-1" /> bKash</>}
                      name="paymentMethod"
                      id="bkash"
                      checked={paymentMethod === 'bkash'}
                      onChange={() => setPaymentMethod('bkash')}
                    />
                    <Form.Check
                      inline
                      type="radio"
                      label={<><FaMobileAlt className="me-1" /> Nagad</>}
                      name="paymentMethod"
                      id="nagad"
                      checked={paymentMethod === 'nagad'}
                      onChange={() => setPaymentMethod('nagad')}
                    />
                  </div>
                </Form.Group>
                
                {paymentMethod === 'cash' && (
                  <>
                    <Form.Group className="mb-3">
                      <Form.Label>Amount Paid</Form.Label>
                      <InputGroup>
                        <InputGroup.Text>৳</InputGroup.Text>
                        <Form.Control 
                          type="number" 
                          value={amountPaid}
                          onChange={(e) => setAmountPaid(Number(e.target.value))}
                          min={total}
                        />
                      </InputGroup>
                    </Form.Group>
                    
                    <Form.Group className="mb-3">
                      <Form.Label>Change</Form.Label>
                      <InputGroup>
                        <InputGroup.Text>৳</InputGroup.Text>
                        <Form.Control 
                          type="number" 
                          value={change}
                          readOnly
                        />
                      </InputGroup>
                    </Form.Group>
                  </>
                )}
                
                {(paymentMethod === 'bkash' || paymentMethod === 'nagad') && (
                  <Form.Group className="mb-3">
                    <Form.Label>Transaction ID</Form.Label>
                    <Form.Control 
                      type="text" 
                      value={transactionId}
                      onChange={(e) => setTransactionId(e.target.value)}
                      placeholder="Enter transaction ID"
                    />
                  </Form.Group>
                )}
              </Form>
            </Col>
          </Row>
        </Modal.Body>
        <Modal.Footer>
          <Button variant="secondary" onClick={() => setShowCheckout(false)}>
            Cancel
          </Button>
          <Button variant="primary" onClick={() => handleCheckout()}>
            <FaSave className="me-1" /> Complete Sale
          </Button>
          <Button variant="info">
            <FaPrint className="me-1" /> Print Invoice
          </Button>
        </Modal.Footer>
      </Modal>

      {/* DSO Stock Selection Modal */}
      <DSOStockSelectionModal
        show={showDSOStockModal}
        onHide={() => setShowDSOStockModal(false)}
        cartItems={cart}
        onStockSelectionComplete={handleDSOStockSelectionComplete}
      />
    </div>
  );
};

export default POS;