import axios from 'axios';

class BkashSmsService {
  constructor() {
    this.baseURL = 'http://localhost:3002';
    this.apiClient = axios.create({
      baseURL: this.baseURL,
      timeout: 10000,
      headers: {
        'Content-Type': 'application/json',
      },
    });

    // Add request interceptor for logging
    this.apiClient.interceptors.request.use(
      (config) => {
        console.log(`[BkashSMS API] ${config.method?.toUpperCase()} ${config.url}`);
        return config;
      },
      (error) => {
        console.error('[BkashSMS API] Request error:', error);
        return Promise.reject(error);
      }
    );

    // Add response interceptor for error handling
    this.apiClient.interceptors.response.use(
      (response) => {
        console.log(`[BkashSMS API] Response:`, response.status, response.data);
        return response;
      },
      (error) => {
        console.error('[BkashSMS API] Response error:', error.response?.data || error.message);
        return Promise.reject(this.handleError(error));
      }
    );
  }

  handleError(error) {
    if (error.code === 'ECONNREFUSED' || error.code === 'ERR_NETWORK') {
      return {
        message: 'Cannot connect to bKash SMS server. Please ensure the server is running on localhost:3002.',
        type: 'connection_error',
        originalError: error
      };
    }

    if (error.response) {
      return {
        message: error.response.data?.message || 'Server error occurred',
        status: error.response.status,
        type: 'server_error',
        originalError: error
      };
    }

    return {
      message: error.message || 'Unknown error occurred',
      type: 'unknown_error',
      originalError: error
    };
  }

  // Check server health
  async checkHealth() {
    try {
      const response = await this.apiClient.get('/health');
      return {
        success: true,
        data: response.data,
        status: 'connected'
      };
    } catch (error) {
      return {
        success: false,
        error: error,
        status: 'disconnected'
      };
    }
  }

  // Get server information
  async getServerInfo() {
    try {
      const response = await this.apiClient.get('/');
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Get all SMS transactions
  async getTransactions(params = {}) {
    try {
      const response = await this.apiClient.get('/api/bkash-sms/transactions', { params });
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Get transactions with pagination
  async getTransactionsPaginated(page = 1, limit = 50, filters = {}) {
    try {
      const params = {
        page,
        limit,
        ...filters
      };
      
      const response = await this.apiClient.get('/api/bkash-sms/transactions', { params });
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Get transaction by ID
  async getTransactionById(id) {
    try {
      const response = await this.apiClient.get(`/api/bkash-sms/transactions/${id}`);
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Get transaction statistics
  async getTransactionStats(period = 'today') {
    try {
      const response = await this.apiClient.get('/api/bkash-sms/stats', {
        params: { period }
      });
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Search transactions
  async searchTransactions(query, filters = {}) {
    try {
      const params = {
        q: query,
        ...filters
      };
      
      const response = await this.apiClient.get('/api/bkash-sms/search', { params });
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Manually add SMS data (for testing)
  async addSmsData(smsData) {
    try {
      const response = await this.apiClient.post('/api/bkash-sms/receive', smsData);
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Update transaction status
  async updateTransactionStatus(id, status) {
    try {
      const response = await this.apiClient.patch(`/api/bkash-sms/transactions/${id}`, {
        status
      });
      return {
        success: true,
        data: response.data
      };
    } catch (error) {
      return {
        success: false,
        error: error
      };
    }
  }

  // Get transaction types
  getTransactionTypes() {
    return [
      { value: 'cash_in', label: 'Cash In', color: 'success' },
      { value: 'cash_out', label: 'Cash Out', color: 'danger' },
      { value: 'send_money', label: 'Send Money', color: 'primary' },
      { value: 'payment', label: 'Payment', color: 'info' },
      { value: 'mobile_recharge', label: 'Mobile Recharge', color: 'warning' }
    ];
  }

  // Get transaction statuses
  getTransactionStatuses() {
    return [
      { value: 'pending', label: 'Pending', color: 'warning' },
      { value: 'processed', label: 'Processed', color: 'success' },
      { value: 'failed', label: 'Failed', color: 'danger' },
      { value: 'duplicate', label: 'Duplicate', color: 'secondary' }
    ];
  }

  // Format currency
  formatCurrency(amount) {
    return new Intl.NumberFormat('en-BD', {
      style: 'currency',
      currency: 'BDT',
      minimumFractionDigits: 2
    }).format(amount);
  }

  // Format date
  formatDate(dateString) {
    return new Date(dateString).toLocaleString('en-BD', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  }

  // Validate phone number
  isValidBangladeshiPhoneNumber(phoneNumber) {
    const cleanNumber = phoneNumber.replace(/\D/g, '');
    return /^(?:\+?88)?01[3-9]\d{8}$/.test(cleanNumber);
  }

  // Extract phone number
  extractPhoneNumber(text) {
    const phoneRegex = /(?:\+?88)?01[3-9]\d{8}/g;
    const matches = text.match(phoneRegex);
    return matches ? matches[0] : null;
  }

  // Get connection status
  async getConnectionStatus() {
    const health = await this.checkHealth();
    return {
      connected: health.success,
      status: health.status,
      serverUrl: this.baseURL,
      lastChecked: new Date().toISOString()
    };
  }
}

// Create and export a singleton instance
const bkashSmsService = new BkashSmsService();
export default bkashSmsService;